<?php

namespace App\Filters;

use CodeIgniter\Filters\FilterInterface;
use CodeIgniter\HTTP\RequestInterface;
use CodeIgniter\HTTP\ResponseInterface;

/**
 * ============================================================================
 * JWT ADMIN FILTER
 * ============================================================================
 */
class JWTAdminFilter implements FilterInterface
{
    public function before(RequestInterface $request, $arguments = null)
    {
        $user = $request->jwt_user ?? null;

        if (!$user) {
            return $this->errorResponse('Yetkilendirme gerekli', 'unauthorized', 401);
        }

        $userType = $user['type'] ?? 'USER';

        if (!in_array($userType, ['ADMIN', 'SUPER_ADMIN'])) {
            $this->logUnauthorizedAccess($request, $user, 'admin_access_attempt');
            return $this->errorResponse('Admin yetkisi gerekli', 'forbidden', 403);
        }

        return $request;
    }

    public function after(RequestInterface $request, ResponseInterface $response, $arguments = null)
    {
        return $response;
    }

    private function errorResponse(string $message, string $code, int $httpCode): ResponseInterface
    {
        return service('response')
            ->setStatusCode($httpCode)
            ->setJSON(['code' => $code, 'message' => $message]);
    }

    private function logUnauthorizedAccess($request, $user, $eventType): void
    {
        try {
            $db = \Config\Database::connect('logs');
            $db->table('security_logs')->insert([
                'event_type' => $eventType,
                'user_id' => $user['id'] ?? null,
                'ip_address' => $request->getIPAddress(),
                'user_agent' => $request->getUserAgent()->getAgentString(),
                'data' => json_encode([
                    'path' => $request->getPath(),
                    'method' => $request->getMethod(),
                    'user_type' => $user['type'] ?? 'unknown',
                ]),
                'created_at' => date('Y-m-d H:i:s'),
            ]);
        } catch (\Exception $e) {
            log_message('error', 'Security log error: ' . $e->getMessage());
        }
    }
}

/**
 * ============================================================================
 * JWT SUPER ADMIN FILTER
 * ============================================================================
 */
class JWTSuperAdminFilter implements FilterInterface
{
    public function before(RequestInterface $request, $arguments = null)
    {
        $user = $request->jwt_user ?? null;

        if (!$user) {
            return $this->errorResponse('Yetkilendirme gerekli', 'unauthorized', 401);
        }

        $userType = $user['type'] ?? 'USER';

        if ($userType !== 'SUPER_ADMIN') {
            $this->logUnauthorizedAccess($request, $user, 'super_admin_access_attempt');
            return $this->errorResponse('Super Admin yetkisi gerekli', 'forbidden', 403);
        }

        return $request;
    }

    public function after(RequestInterface $request, ResponseInterface $response, $arguments = null)
    {
        return $response;
    }

    private function errorResponse(string $message, string $code, int $httpCode): ResponseInterface
    {
        return service('response')
            ->setStatusCode($httpCode)
            ->setJSON(['code' => $code, 'message' => $message]);
    }

    private function logUnauthorizedAccess($request, $user, $eventType): void
    {
        try {
            $db = \Config\Database::connect('logs');
            $db->table('security_logs')->insert([
                'event_type' => $eventType,
                'user_id' => $user['id'] ?? null,
                'ip_address' => $request->getIPAddress(),
                'user_agent' => $request->getUserAgent()->getAgentString(),
                'data' => json_encode([
                    'path' => $request->getPath(),
                    'method' => $request->getMethod(),
                ]),
                'created_at' => date('Y-m-d H:i:s'),
            ]);
        } catch (\Exception $e) {
            log_message('error', 'Security log error: ' . $e->getMessage());
        }
    }
}

/**
 * ============================================================================
 * JWT AGENT FILTER
 * ============================================================================
 */
class JWTAgentFilter implements FilterInterface
{
    public function before(RequestInterface $request, $arguments = null)
    {
        $user = $request->jwt_user ?? null;

        if (!$user) {
            return $this->errorResponse('Yetkilendirme gerekli', 'unauthorized', 401);
        }

        $userType = $user['type'] ?? 'USER';
        $allowedTypes = ['AGENT', 'ADMIN', 'SUPER_ADMIN'];

        if (!in_array($userType, $allowedTypes)) {
            return $this->errorResponse('Agent yetkisi gerekli', 'forbidden', 403);
        }

        return $request;
    }

    public function after(RequestInterface $request, ResponseInterface $response, $arguments = null)
    {
        return $response;
    }

    private function errorResponse(string $message, string $code, int $httpCode): ResponseInterface
    {
        return service('response')
            ->setStatusCode($httpCode)
            ->setJSON(['code' => $code, 'message' => $message]);
    }
}

/**
 * ============================================================================
 * JWT RATE LIMIT FILTER
 * ============================================================================
 */
class JWTRateLimitFilter implements FilterInterface
{
    private int $maxRequests = 100;
    private int $windowSeconds = 60;

    public function before(RequestInterface $request, $arguments = null)
    {
        $cache = service('cache');
        $ip = $request->getIPAddress();
        $userId = $request->jwt_user['id'] ?? null;
        
        // User varsa user bazlı, yoksa IP bazlı rate limit
        $cacheKey = $userId 
            ? 'rate_limit_user_' . $userId 
            : 'rate_limit_ip_' . md5($ip);

        $current = $cache->get($cacheKey);

        if ($current === null) {
            $cache->save($cacheKey, 1, $this->windowSeconds);
            return $request;
        }

        if ($current >= $this->maxRequests) {
            $this->logRateLimitExceeded($request, $ip, $userId);
            return $this->rateLimitResponse();
        }

        $cache->save($cacheKey, $current + 1, $this->windowSeconds);

        // Rate limit headers
        service('response')
            ->setHeader('X-RateLimit-Limit', (string) $this->maxRequests)
            ->setHeader('X-RateLimit-Remaining', (string) ($this->maxRequests - $current - 1));

        return $request;
    }

    public function after(RequestInterface $request, ResponseInterface $response, $arguments = null)
    {
        return $response;
    }

    private function rateLimitResponse(): ResponseInterface
    {
        return service('response')
            ->setStatusCode(429)
            ->setHeader('Retry-After', (string) $this->windowSeconds)
            ->setJSON([
                'code' => 'rate_limit_exceeded',
                'message' => 'Çok fazla istek. Lütfen bekleyin.',
                'retry_after' => $this->windowSeconds,
            ]);
    }

    private function logRateLimitExceeded($request, $ip, $userId): void
    {
        try {
            $db = \Config\Database::connect('logs');
            $db->table('security_logs')->insert([
                'event_type' => 'rate_limit_exceeded',
                'user_id' => $userId,
                'ip_address' => $ip,
                'user_agent' => $request->getUserAgent()->getAgentString(),
                'data' => json_encode([
                    'path' => $request->getPath(),
                    'method' => $request->getMethod(),
                ]),
                'created_at' => date('Y-m-d H:i:s'),
            ]);
        } catch (\Exception $e) {
            // Sessizce devam et
        }
    }
}

/**
 * ============================================================================
 * EMAIL VERIFIED FILTER
 * ============================================================================
 */
class EmailVerifiedFilter implements FilterInterface
{
    public function before(RequestInterface $request, $arguments = null)
    {
        $user = $request->jwt_user ?? null;

        if (!$user) {
            return $this->errorResponse('Yetkilendirme gerekli', 'unauthorized', 401);
        }

        $verified = $user['verified'] ?? false;

        if (!$verified) {
            return $this->errorResponse(
                'E-posta adresinizi doğrulamanız gerekiyor',
                'email_not_verified',
                403
            );
        }

        return $request;
    }

    public function after(RequestInterface $request, ResponseInterface $response, $arguments = null)
    {
        return $response;
    }

    private function errorResponse(string $message, string $code, int $httpCode): ResponseInterface
    {
        return service('response')
            ->setStatusCode($httpCode)
            ->setJSON(['code' => $code, 'message' => $message]);
    }
}

/**
 * ============================================================================
 * CRON AUTH FILTER
 * ============================================================================
 */
class CronAuthFilter implements FilterInterface
{
    // İzin verilen IP'ler
    private array $allowedIPs = [
        '127.0.0.1',
        '::1',
        // Sunucu IP'nizi ekleyin
    ];

    public function before(RequestInterface $request, $arguments = null)
    {
        $ip = $request->getIPAddress();

        // IP kontrolü
        if (in_array($ip, $this->allowedIPs)) {
            return $request;
        }

        // Cron secret key kontrolü
        $cronKey = $request->getHeaderLine('X-Cron-Key');
        $expectedKey = getenv('CRON_SECRET_KEY') ?: '';

        if (!empty($expectedKey) && $cronKey === $expectedKey) {
            return $request;
        }

        log_message('warning', "Unauthorized cron access attempt from IP: {$ip}");
        
        return service('response')
            ->setStatusCode(403)
            ->setJSON(['code' => 'forbidden', 'message' => 'Cron erişimi engellendi']);
    }

    public function after(RequestInterface $request, ResponseInterface $response, $arguments = null)
    {
        return $response;
    }
}

/**
 * ============================================================================
 * WEBHOOK AUTH FILTER
 * ============================================================================
 */
class WebhookAuthFilter implements FilterInterface
{
    public function before(RequestInterface $request, $arguments = null)
    {
        $webhookSecret = getenv('WEBHOOK_SECRET') ?: '';

        if (empty($webhookSecret)) {
            return $request; // Secret tanımlı değilse geç
        }

        // Signature doğrulama
        $signature = $request->getHeaderLine('X-Webhook-Signature');
        $body = $request->getBody();
        $expectedSignature = hash_hmac('sha256', $body, $webhookSecret);

        if (!hash_equals($expectedSignature, $signature)) {
            log_message('warning', 'Invalid webhook signature');
            return service('response')
                ->setStatusCode(403)
                ->setJSON(['code' => 'forbidden', 'message' => 'Webhook doğrulama hatası']);
        }

        return $request;
    }

    public function after(RequestInterface $request, ResponseInterface $response, $arguments = null)
    {
        return $response;
    }
}
