<?php

namespace App\Libraries;

/**
 * ============================================================================
 * MANUEL JWT LIBRARY
 * ============================================================================
 * 
 * Composer gerektirmeyen, sıfırdan yazılmış JWT kütüphanesi
 * Shared hosting uyumlu
 * 
 * @package    Satisahazir
 * @version    2.0.0
 * ============================================================================
 */

// JWT Exceptions
class TokenExpiredException extends \Exception {}
class TokenInvalidException extends \Exception {}
class SignatureInvalidException extends \Exception {}

/**
 * JWT Class
 */
class JWT
{
    private string $algorithm = 'HS256';
    
    private array $supportedAlgorithms = [
        'HS256' => 'sha256',
        'HS384' => 'sha384',
        'HS512' => 'sha512',
    ];

    /**
     * Token encode
     */
    public function encode(array $payload, string $key, string $algorithm = 'HS256'): string
    {
        $header = [
            'typ' => 'JWT',
            'alg' => $algorithm,
        ];

        $segments = [];
        $segments[] = $this->base64UrlEncode(json_encode($header));
        $segments[] = $this->base64UrlEncode(json_encode($payload));

        $signingInput = implode('.', $segments);
        $signature = $this->sign($signingInput, $key, $algorithm);
        $segments[] = $this->base64UrlEncode($signature);

        return implode('.', $segments);
    }

    /**
     * Token decode
     */
    public function decode(string $token, string $key, array $options = []): array
    {
        $leeway = $options['leeway'] ?? 0;

        $parts = explode('.', $token);
        if (count($parts) !== 3) {
            throw new TokenInvalidException('Token format geçersiz');
        }

        [$headerB64, $payloadB64, $signatureB64] = $parts;

        // Header decode
        $header = json_decode($this->base64UrlDecode($headerB64), true);
        if (!$header) {
            throw new TokenInvalidException('Header decode edilemedi');
        }

        // Payload decode
        $payload = json_decode($this->base64UrlDecode($payloadB64), true);
        if (!$payload) {
            throw new TokenInvalidException('Payload decode edilemedi');
        }

        // Signature decode
        $signature = $this->base64UrlDecode($signatureB64);

        // Algorithm check
        $algorithm = $header['alg'] ?? 'HS256';
        if (!isset($this->supportedAlgorithms[$algorithm])) {
            throw new TokenInvalidException('Desteklenmeyen algoritma: ' . $algorithm);
        }

        // Signature verify
        $signingInput = $headerB64 . '.' . $payloadB64;
        $expectedSignature = $this->sign($signingInput, $key, $algorithm);

        if (!hash_equals($expectedSignature, $signature)) {
            throw new SignatureInvalidException('Token imzası geçersiz');
        }

        // Expiration check
        if (isset($payload['exp'])) {
            if (($payload['exp'] + $leeway) < time()) {
                throw new TokenExpiredException('Token süresi dolmuş');
            }
        }

        // Not before check
        if (isset($payload['nbf'])) {
            if (($payload['nbf'] - $leeway) > time()) {
                throw new TokenInvalidException('Token henüz geçerli değil');
            }
        }

        // Issued at check
        if (isset($payload['iat'])) {
            if (($payload['iat'] - $leeway) > time()) {
                throw new TokenInvalidException('Token gelecekte oluşturulmuş');
            }
        }

        return $payload;
    }

    /**
     * Token doğrula (exception fırlatmadan)
     */
    public function validate(string $token, string $key): ?array
    {
        try {
            return $this->decode($token, $key);
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Token'dan payload al (doğrulama olmadan)
     */
    public function getPayloadWithoutVerification(string $token): ?array
    {
        $parts = explode('.', $token);
        if (count($parts) !== 3) {
            return null;
        }

        $payload = json_decode($this->base64UrlDecode($parts[1]), true);
        return $payload ?: null;
    }

    /**
     * Sign
     */
    private function sign(string $input, string $key, string $algorithm): string
    {
        $hashAlgorithm = $this->supportedAlgorithms[$algorithm] ?? 'sha256';
        return hash_hmac($hashAlgorithm, $input, $key, true);
    }

    /**
     * Base64 URL encode
     */
    private function base64UrlEncode(string $data): string
    {
        return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
    }

    /**
     * Base64 URL decode
     */
    private function base64UrlDecode(string $data): string
    {
        $remainder = strlen($data) % 4;
        if ($remainder) {
            $data .= str_repeat('=', 4 - $remainder);
        }
        return base64_decode(strtr($data, '-_', '+/'));
    }
}
