<?php

namespace App\Models;

use CodeIgniter\Model;

class UserModel extends Model
{
    protected $table      = 'user';
    protected $primaryKey = 'id';

    protected $useAutoIncrement = true;

    protected $returnType     = 'array';
    protected $useSoftDeletes = false;

    // E-posta doğrulama alanları eklendi
protected $allowedFields = [
    'username', 
    'phone', 
    'gplus_link', 
    'facebook_link', 
    'mail', 
    'name_surname', 
    'address', 
    'zip_city', 
    'password',
    'auth_gmail',
    'auth_uid',
    'type',
    'language',
    'activated',
    'registration_date',
    'last_login',
    'fcmToken',
    'mail_verified',
    'phone_verified',
    // E-POSTA DOĞRULAMA ALANLARI (mevcut)
    'verification_code',
    'verification_token', 
    'verification_sent_at',
    'verification_expires_at',
    'email_verified_at',
    // YENİ: ŞİFRE SIFIRLAMA ALANLARI
    'reset_code',
    'reset_code_sent_at', 
    'reset_code_expires_at',
    'reset_code_verified',
    'reset_code_verified_at',
    'password_updated_at',
    // DİĞER
    'image_user_filename'
];

    // Custom vars
    protected $orderBy = 'name_surname ASC';

    // Dates
    protected $useTimestamps = false;
    protected $dateFormat    = 'datetime';
    protected $createdField  = 'created_at';
    protected $updatedField  = 'updated_at';
    protected $deletedField  = 'deleted_at';

    // Validation
    protected $validationRules      = [];
    protected $validationMessages   = [];
    protected $skipValidation       = true;
    protected $cleanValidationRules = true;

    // Callbacks
    protected $allowCallbacks = true;
    protected $beforeInsert   = [];
    protected $afterInsert    = [];
    protected $beforeUpdate   = [];
    protected $afterUpdate    = [];
    protected $beforeFind     = [];
    protected $afterFind      = [];
    protected $beforeDelete   = [];
    protected $afterDelete    = [];

    // =======================================================
    // TEMEL METODLAR
    // =======================================================

    public function findAll($limit = 0, $offset = 0)
    {
        $results = parent::findAll($limit, $offset);
        
        return $results;
    }

    public function findAllListing($limit = 0, $offset = 0)
    {
        $this->select($this->table.'.*, property_user.property_id, COUNT(user_id) as count_listings');
        $this->join('property_user', $this->table.'.id = property_user.user_id', 'left');
        $this->groupBy('user_id');
        $this->orderBy('count_listings', 'DESC');

        $results = parent::findAll($limit, $offset);
        
        return $results;
    }

    public function get_pagination($limit, $offset, $where = array(), $user_check = FALSE, $user_id=NULL, $show_other_agents_litings = FALSE)
    {
        $fields_table = '';
        $results = $this->findAll($limit, $offset);
        
        return $results;
    }

    // =======================================================
    // ŞİFRELEME VE HASH METODLARI
    // =======================================================

    public function hash($string)
    {
       if(getenv('hash_function') == '')
       {
           if (function_exists('hash')) {
                return substr(hash('sha512', $string.getenv('encryption_key')), 0, 10);
           }
    
           return substr(md5($string.getenv('encryption_key')), 0, 10);
       }
       else if(getenv('hash_function') == 'hash')
       {
            return substr(hash('sha512', $string.getenv('encryption_key')), 0, 10);
       }
       else if(getenv('hash_function') == 'md5')
       {
            return substr(md5($string.getenv('encryption_key')), 0, 10);
       }
    }
    
    public function hash_token($string)
    {
       if(getenv('hash_function') == '')
       {
           if (function_exists('hash')) {
                return substr(hash('sha512', $string.getenv('encryption_key')), 0, 32);
           }
    
           return substr(md5($string.getenv('encryption_key')), 0, 32);
       }
       else if(getenv('hash_function') == 'hash')
       {
            return substr(hash('sha512', $string.getenv('encryption_key')), 0, 32);
       }
       else if(getenv('hash_function') == 'md5')
       {
            return substr(md5($string.getenv('encryption_key')), 0, 32);
       }
    }

    // =======================================================
    // KULLANICI SİLME METODLARI
    // =======================================================
/*
    public function delete_item($id) {
        // İlişkili verileri sil
        $builder = $this->db->table('property_user');
        $builder->where('user_id', $id);
        $builder->delete();

        $builder = $this->db->table('reservations');
        $builder->where('user_id', $id);
        $builder->delete();

        $builder = $this->db->table('favorites');
        $builder->where('user_id', $id);
        $builder->delete();

        $builder = $this->db->table('saved_search');
        $builder->where('user_id', $id);
        $builder->delete();

        $builder = $this->db->table('affilate_packages');
        $builder->where('user_id', $id);
        $builder->delete();

        $builder = $this->db->table('token_api');
        $builder->where('user_id', $id);
        $builder->delete();

        // Kullanıcıyı tamamen sil (e-posta doğrulama verileri de dahil)
        $builder = $this->db->table('user');
        $builder->where('id', $id);
        return $builder->delete();
    }*/

    // =======================================================
    // KOLAY KOD SİSTEMİ METODLARI
    // =======================================================

    /**
     * Kolay hatırlanabilir 6 haneli kod oluştur
     * İlk 3 rakam = Son 3 rakam (örnek: 654654, 321321)
     */
    public function generateEasyVerificationCode()
    {
        // 100-999 arası rastgele 3 haneli sayı
        $firstThree = sprintf('%03d', mt_rand(100, 999));
        
        // Aynı 3 rakamı tekrarla
        $verificationCode = $firstThree . $firstThree;
        
        return $verificationCode;
    }

    /**
     * Kolay kod formatında mı kontrol et (654654 formatı)
     */
    public function isEasyCodeFormat($code)
    {
        // 6 haneli olmalı ve sadece rakam içermeli
        if (!preg_match('/^\d{6}$/', $code)) {
            return false;
        }

        // İlk 3 hane = Son 3 hane olmalı
        $firstThree = substr($code, 0, 3);
        $lastThree = substr($code, 3, 3);

        return $firstThree === $lastThree;
    }

    // =======================================================
    // E-POSTA DOĞRULAMA VERİ TEMİZLEME METODLARI
    // =======================================================

    /**
     * Sadece e-posta doğrulama verilerini temizle (kullanıcıyı silmeden)
     */
    public function clearVerificationData($userId) {
        return $this->update($userId, [
            'verification_code' => null,
            'verification_token' => null,
            'verification_sent_at' => null,
            'verification_expires_at' => null
        ]);
    }

    /**
     * Süresi dolmuş doğrulama verilerini temizle
     */
    public function cleanExpiredVerifications() {
        return $this->where('verification_expires_at <', date('Y-m-d H:i:s'))
                   ->set([
                       'verification_code' => null,
                       'verification_token' => null,
                       'verification_expires_at' => null
                   ])
                   ->update();
    }

    // =======================================================
    // E-POSTA DOĞRULAMA ARAMA METODLARI
    // =======================================================

    /**
     * E-posta adresine göre kullanıcıyı bul
     */
    public function findByEmail($email)
    {
        return $this->where('mail', $email)->first();
    }

    /**
     * Doğrulama token'ına göre kullanıcıyı bul
     */
    public function findByVerificationToken($token)
    {
        return $this->where('verification_token', $token)->first();
    }

    /**
     * E-posta ve doğrulama koduna göre kullanıcıyı bul
     */
    public function findByEmailAndCode($email, $code)
    {
        return $this->where('mail', $email)
                   ->where('verification_code', $code)
                   ->first();
    }

    // =======================================================
    // E-POSTA DOĞRULAMA İŞLEMLERİ
    // =======================================================

    /**
     * Kullanıcıyı e-posta doğrulanmış olarak işaretle
     */
    public function markEmailAsVerified($userId)
    {
        return $this->update($userId, [
            'mail_verified' => 1,
            'verification_token' => null,
            'verification_code' => null,
            'verification_expires_at' => null,
            'email_verified_at' => date('Y-m-d H:i:s')
        ]);
    }

    /**
     * Kolay doğrulama kodu ve token oluştur
     */
    public function generateVerificationData($userId)
    {
        // Kolay kod sistemi kullan
        $verificationCode = $this->generateEasyVerificationCode();
        $verificationToken = bin2hex(random_bytes(32));
        
        $updateData = [
            'verification_code' => $verificationCode,
            'verification_token' => $verificationToken,
            'verification_sent_at' => date('Y-m-d H:i:s'),
            'verification_expires_at' => date('Y-m-d H:i:s', strtotime('+24 hours'))
        ];

        $this->update($userId, $updateData);

        return [
            'code' => $verificationCode,
            'token' => $verificationToken,
            'expires_at' => $updateData['verification_expires_at']
        ];
    }

    /**
     * Doğrulama kodu ile kullanıcı doğrula
     */
    public function verifyEmailWithCode($email, $code)
    {
        // Kullanıcıyı bul
        $user = $this->where('mail', $email)
                    ->where('verification_code', $code)
                    ->first();

        if (!$user) {
            return ['success' => false, 'message' => 'Geçersiz doğrulama kodu'];
        }

        // Süre kontrolü
        if (strtotime($user['verification_expires_at']) < time()) {
            return ['success' => false, 'message' => 'Doğrulama kodu süresi dolmuş'];
        }

        // Zaten doğrulanmış mı
        if ($user['mail_verified'] == 1) {
            return ['success' => true, 'message' => 'E-posta zaten doğrulanmış', 'already_verified' => true];
        }

        // Doğrulamayı gerçekleştir
        $updateResult = $this->markEmailAsVerified($user['id']);

        if ($updateResult) {
            return [
                'success' => true, 
                'message' => 'E-posta başarıyla doğrulandı',
                'user_id' => $user['id']
            ];
        } else {
            return ['success' => false, 'message' => 'Doğrulama güncellenirken hata oluştu'];
        }
    }

    /**
     * Doğrulama kodunu yenile
     */
    public function refreshVerificationCode($email)
    {
        $user = $this->where('mail', $email)->first();
        
        if (!$user) {
            return ['success' => false, 'message' => 'Kullanıcı bulunamadı'];
        }

        // Zaten doğrulanmış mı
        if ($user['mail_verified'] == 1) {
            return ['success' => false, 'message' => 'E-posta zaten doğrulanmış'];
        }

        // Spam kontrolü
        $canSend = $this->canSendVerificationByEmail($email, 1);
        if (!$canSend['can_send']) {
            return [
                'success' => false, 
                'message' => "Lütfen {$canSend['wait_time']} saniye bekleyip tekrar deneyin"
            ];
        }

        // Yeni kod oluştur
        $verificationData = $this->generateVerificationData($user['id']);

        return [
            'success' => true,
            'message' => 'Yeni doğrulama kodu gönderildi',
            'code' => $verificationData['code'],
            'expires_at' => $verificationData['expires_at']
        ];
    }

    // =======================================================
    // E-POSTA DOĞRULAMA DURUM KONTROLLERI
    // =======================================================

    /**
     * E-posta doğrulama durumunu kontrol et
     */
    public function isEmailVerified($userId)
    {
        $user = $this->find($userId);
        return $user && $user['mail_verified'] == 1;
    }

    /**
     * Son doğrulama gönderim zamanını kontrol et (spam koruması)
     */
    public function canSendVerification($userId, $waitTimeMinutes = 1)
    {
        $user = $this->find($userId);
        
        if (!$user || empty($user['verification_sent_at'])) {
            return true;
        }

        $lastSent = strtotime($user['verification_sent_at']);
        $now = time();
        $waitTime = $waitTimeMinutes * 60; // dakikayı saniyeye çevir

        return ($now - $lastSent) >= $waitTime;
    }

    /**
     * E-posta adresine göre son doğrulama gönderim zamanını kontrol et
     */
    public function canSendVerificationByEmail($email, $waitTimeMinutes = 1)
    {
        $user = $this->where('mail', $email)->first();
        
        if (!$user || empty($user['verification_sent_at'])) {
            return ['can_send' => true, 'wait_time' => 0];
        }

        $lastSent = strtotime($user['verification_sent_at']);
        $now = time();
        $waitTime = $waitTimeMinutes * 60;
        $timePassed = $now - $lastSent;

        if ($timePassed >= $waitTime) {
            return ['can_send' => true, 'wait_time' => 0];
        } else {
            $remainingWait = $waitTime - $timePassed;
            return ['can_send' => false, 'wait_time' => $remainingWait];
        }
    }

    /**
     * Doğrulama kodunun süresi dolmuş mu kontrol et
     */
    public function isVerificationExpired($userId)
    {
        $user = $this->find($userId);
        
        if (!$user || empty($user['verification_expires_at'])) {
            return true;
        }

        return strtotime($user['verification_expires_at']) < time();
    }

    /**
     * E-posta adresine göre doğrulama kodunun süresi dolmuş mu kontrol et
     */
    public function isVerificationExpiredByEmail($email)
    {
        $user = $this->where('mail', $email)->first();
        
        if (!$user || empty($user['verification_expires_at'])) {
            return true;
        }

        return strtotime($user['verification_expires_at']) < time();
    }

    /**
     * Kullanıcının aktif doğrulama kodu var mı kontrol et
     */
    public function hasActiveVerificationCode($email)
    {
        $user = $this->where('mail', $email)->first();
        
        if (!$user || empty($user['verification_code']) || empty($user['verification_expires_at'])) {
            return false;
        }

        // Süre kontrolü
        return strtotime($user['verification_expires_at']) > time();
    }

    // =======================================================
    // ESKİ UYUMLULUK METODLARI (GERİYE DÖNÜK UYUMLULUK İÇİN)
    // =======================================================

    /**
     * ESKİ: Rastgele 6 haneli kod oluştur (geriye dönük uyumluluk için)
     * Artık generateEasyVerificationCode() kullanılmalı
     */
    public function generateVerificationCode()
    {
        // Kolay kod sistemini kullan
        return $this->generateEasyVerificationCode();
    }

    // =======================================================
    // YARDIMCI METODLAR
    // =======================================================

    /**
     * Kullanıcının toplam e-posta doğrulama deneme sayısını getir
     */
    public function getVerificationAttempts($email, $hours = 24)
    {
        $user = $this->where('mail', $email)->first();
        
        if (!$user) {
            return 0;
        }

        // Son X saat içindeki deneme sayısını hesapla
        // Bu özellik için ayrı bir log tablosu gerekebilir
        // Şimdilik basit bir kontrol yapıyoruz
        
        if (empty($user['verification_sent_at'])) {
            return 0;
        }

        $lastSent = strtotime($user['verification_sent_at']);
        $timeAgo = time() - ($hours * 3600);

        return $lastSent > $timeAgo ? 1 : 0;
    }

    /**
     * Kullanıcının e-posta doğrulama geçmişini temizle
     */
    public function resetVerificationHistory($userId)
    {
        return $this->clearVerificationData($userId);
    }

    /**
     * Doğrulama kodu debug bilgisi (sadece development için)
     */
    public function getVerificationDebugInfo($email)
    {
        if (ENVIRONMENT !== 'development') {
            return ['error' => 'Bu özellik sadece development ortamında kullanılabilir'];
        }

        $user = $this->where('mail', $email)->first();
        
        if (!$user) {
            return ['error' => 'Kullanıcı bulunamadı'];
        }

        return [
            'user_id' => $user['id'],
            'email' => $user['mail'],
            'verified' => $user['mail_verified'] == 1,
            'current_code' => $user['verification_code'] ?? 'Yok',
            'sent_at' => $user['verification_sent_at'] ?? 'Hiçbir zaman',
            'expires_at' => $user['verification_expires_at'] ?? 'Süre yok',
            'is_expired' => $this->isVerificationExpiredByEmail($email),
            'can_send_new' => $this->canSendVerificationByEmail($email, 1)['can_send'],
        ];
    }
     // =======================================================
    // ŞİFRE SIFIRLAMA METODLARI
    // =======================================================

    /**
     * Şifre sıfırlama kodu oluştur ve kaydet
     */
    public function generatePasswordResetCode($email)
    {
        $user = $this->findByEmail($email);
        
        if (!$user) {
            return ['success' => false, 'message' => 'Kullanıcı bulunamadı'];
        }

        // E-posta doğrulanmış mı kontrol et
        if ($user['mail_verified'] != 1) {
            return ['success' => false, 'message' => 'E-posta adresi doğrulanmamış'];
        }

        // Rate limiting - Son gönderimden 60 saniye geçmiş mi?
        if (!empty($user['reset_code_sent_at'])) {
            $lastSent = strtotime($user['reset_code_sent_at']);
            $now = time();
            if (($now - $lastSent) < 60) {
                $waitTime = 60 - ($now - $lastSent);
                return [
                    'success' => false, 
                    'message' => "Lütfen {$waitTime} saniye bekleyip tekrar deneyin",
                    'wait_time' => $waitTime
                ];
            }
        }

        // Kolay kod sistemi kullan
        $resetCode = $this->generateEasyVerificationCode();
        
        $updateData = [
            'reset_code' => $resetCode,
            'reset_code_sent_at' => date('Y-m-d H:i:s'),
            'reset_code_expires_at' => date('Y-m-d H:i:s', strtotime('+30 minutes')), // 30 dakika geçerli
            'reset_code_verified' => 0,
            'reset_code_verified_at' => null
        ];

        $updateResult = $this->update($user['id'], $updateData);

        if ($updateResult) {
            return [
                'success' => true,
                'message' => 'Sıfırlama kodu oluşturuldu',
                'code' => $resetCode,
                'expires_at' => $updateData['reset_code_expires_at'],
                'user_id' => $user['id']
            ];
        } else {
            return ['success' => false, 'message' => 'Kod oluşturulurken hata oluştu'];
        }
    }

    /**
     * Şifre sıfırlama kodunu doğrula
     */
    public function verifyPasswordResetCode($email, $code)
    {
        $user = $this->where('mail', $email)
                    ->where('reset_code', $code)
                    ->first();

        if (!$user) {
            return ['success' => false, 'message' => 'Geçersiz sıfırlama kodu'];
        }

        // Kodun süresi dolmuş mu?
        if (empty($user['reset_code_expires_at']) || strtotime($user['reset_code_expires_at']) < time()) {
            return ['success' => false, 'message' => 'Sıfırlama kodu süresi dolmuş'];
        }

        // Kodu doğrulanmış olarak işaretle
        $updateData = [
            'reset_code_verified' => 1,
            'reset_code_verified_at' => date('Y-m-d H:i:s')
        ];

        $updateResult = $this->update($user['id'], $updateData);

        if ($updateResult) {
            return [
                'success' => true,
                'message' => 'Sıfırlama kodu doğrulandı',
                'user_id' => $user['id']
            ];
        } else {
            return ['success' => false, 'message' => 'Kod doğrulanırken hata oluştu'];
        }
    }

    /**
     * Şifreyi sıfırla (yeni şifre belirle)
     */
    public function resetPasswordWithCode($email, $code, $newPassword)
    {
        // Şifre uzunluk kontrolü
        if (strlen($newPassword) < 6) {
            return ['success' => false, 'message' => 'Şifre en az 6 karakter olmalıdır'];
        }

        $user = $this->where('mail', $email)
                    ->where('reset_code', $code)
                    ->first();

        if (!$user) {
            return ['success' => false, 'message' => 'Geçersiz sıfırlama kodu'];
        }

        // Kodun süresi dolmuş mu?
        if (empty($user['reset_code_expires_at']) || strtotime($user['reset_code_expires_at']) < time()) {
            return ['success' => false, 'message' => 'Sıfırlama kodu süresi dolmuş'];
        }

        // Kod doğrulanmış mı? (Extra güvenlik)
        if (empty($user['reset_code_verified']) || $user['reset_code_verified'] != 1) {
            return ['success' => false, 'message' => 'Önce sıfırlama kodunu doğrulamanız gerekiyor'];
        }

        // Şifreyi hashle ve güncelle
        $hashedPassword = $this->hash($newPassword);
        
        $updateData = [
            'password' => $hashedPassword,
            'reset_code' => null,
            'reset_code_sent_at' => null,
            'reset_code_expires_at' => null,
            'reset_code_verified' => 0,
            'reset_code_verified_at' => null,
            'password_updated_at' => date('Y-m-d H:i:s'),
            'last_password_change' => date('Y-m-d H:i:s') // Mevcut sütununuz varsa
        ];

        $updateResult = $this->update($user['id'], $updateData);

        if ($updateResult) {
            // Güvenlik için tüm tokenları sil
            $this->clearUserTokens($user['id']);

            return [
                'success' => true,
                'message' => 'Şifre başariyla değiştirildi',
                'user_id' => $user['id']
            ];
        } else {
            return ['success' => false, 'message' => 'Şifre değiştirilirken hata oluştu'];
        }
    }

    /**
     * Kullanıcının tüm tokenlarını sil (güvenlik için)
     */
    private function clearUserTokens($userId)
    {
        try {
            $tokenBuilder = $this->db->table('token_api');
            $tokenBuilder->where('user_id', $userId);
            $tokenBuilder->delete();
            
            log_message('info', "User ID {$userId} - Tüm tokenlar temizlendi (şifre değişikliği)");
        } catch (\Exception $e) {
            log_message('warning', 'Token temizleme hatası: ' . $e->getMessage());
        }
    }

    /**
     * Süresi dolmuş şifre sıfırlama kodlarını temizle
     */
    public function cleanupExpiredResetCodes()
    {
        $cleanupResult = $this->where('reset_code_expires_at <', date('Y-m-d H:i:s'))
                             ->where('reset_code IS NOT NULL')
                             ->set([
                                'reset_code' => null,
                                'reset_code_sent_at' => null,
                                'reset_code_expires_at' => null,
                                'reset_code_verified' => 0,
                                'reset_code_verified_at' => null
                             ])
                             ->update();
        
        $affectedRows = $this->db->affected_rows();
        
        if ($affectedRows > 0) {
            log_message('info', "Temizlenen süresi dolmuş şifre sıfırlama kodu: {$affectedRows}");
        }
        
        return $affectedRows;
    }

    /**
     * E-posta ile şifre sıfırlama durumunu kontrol et
     */
    public function getPasswordResetStatus($email)
    {
        $user = $this->where('mail', $email)->first();
        
        if (!$user) {
            return ['exists' => false];
        }

        return [
            'exists' => true,
            'has_active_code' => !empty($user['reset_code']) && !empty($user['reset_code_expires_at']) && strtotime($user['reset_code_expires_at']) > time(),
            'code_verified' => $user['reset_code_verified'] == 1,
            'last_sent' => $user['reset_code_sent_at'],
            'expires_at' => $user['reset_code_expires_at'],
            'can_send_new' => $this->canSendPasswordReset($email)
        ];
    }

    /**
     * Yeni şifre sıfırlama kodu gönderilip gönderilemeyeceğini kontrol et
     */
    public function canSendPasswordReset($email, $waitTimeMinutes = 1)
    {
        $user = $this->where('mail', $email)->first();
        
        if (!$user || empty($user['reset_code_sent_at'])) {
            return true;
        }

        $lastSent = strtotime($user['reset_code_sent_at']);
        $now = time();
        $waitTime = $waitTimeMinutes * 60;

        return ($now - $lastSent) >= $waitTime;
    }

    /**
     * Kullanıcının şifre sıfırlama geçmişini temizle
     */
    public function clearPasswordResetHistory($userId)
    {
        return $this->update($userId, [
            'reset_code' => null,
            'reset_code_sent_at' => null,
            'reset_code_expires_at' => null,
            'reset_code_verified' => 0,
            'reset_code_verified_at' => null
        ]);
    }

    /**
     * Son şifre değişiklik zamanını güncelle
     */
    public function updatePasswordChangeTime($userId)
    {
        return $this->update($userId, [
            'password_updated_at' => date('Y-m-d H:i:s'),
            'last_password_change' => date('Y-m-d H:i:s')
        ]);
    }

    /**
     * Debug bilgisi - Şifre sıfırlama durumu (sadece development için)
     */
    public function getPasswordResetDebugInfo($email)
    {
        if (ENVIRONMENT !== 'development') {
            return ['error' => 'Bu özellik sadece development ortamında kullanılabilir'];
        }

        $user = $this->where('mail', $email)->first();
        
        if (!$user) {
            return ['error' => 'Kullanıcı bulunamadı'];
        }

        return [
            'user_id' => $user['id'],
            'email' => $user['mail'],
            'email_verified' => $user['mail_verified'] == 1,
            'reset_code' => $user['reset_code'] ?? 'Yok',
            'code_sent_at' => $user['reset_code_sent_at'] ?? 'Hiçbir zaman',
            'code_expires_at' => $user['reset_code_expires_at'] ?? 'Süre yok',
            'code_verified' => $user['reset_code_verified'] == 1,
            'code_verified_at' => $user['reset_code_verified_at'] ?? 'Doğrulanmamış',
            'is_expired' => empty($user['reset_code_expires_at']) ? true : strtotime($user['reset_code_expires_at']) < time(),
            'can_send_new' => $this->canSendPasswordReset($email, 1),
            'last_password_change' => $user['password_updated_at'] ?? 'Bilinmiyor'
        ];
    }
    
    
}