<?php

namespace App\Commands;

use CodeIgniter\CLI\BaseCommand;
use CodeIgniter\CLI\CLI;

/**
 * Randevu hatırlatıcıları için Cron Job
 * Her dakika çalıştırılmalı: * * * * * php spark appointments:remind
 */
class AppointmentReminder extends BaseCommand
{
    protected $group       = 'Appointments';
    protected $name        = 'appointments:remind';
    protected $description = 'Randevu hatırlatıcılarını gönder';

    public function run(array $params)
    {
        CLI::write('Randevu hatırlatıcıları kontrol ediliyor...', 'yellow');
        
        try {
            $enquireModel = model('App\Models\EnquireModel');
            
            // 15 dakika öncesi hatırlatmaları gönder
            $sentCount = $enquireModel->sendAppointmentReminders();
            
            if ($sentCount > 0) {
                CLI::write("✅ $sentCount adet randevu hatırlatıcısı gönderildi", 'green');
            } else {
                CLI::write('Bekleyen hatırlatıcı bulunamadı', 'blue');
            }
            
            // 5 dakika öncesi hatırlatmaları için de kontrol et
            $this->checkUpcomingAppointments();
            
        } catch (\Exception $e) {
            CLI::write('❌ Hata: ' . $e->getMessage(), 'red');
            log_message('error', 'AppointmentReminder Cron Error: ' . $e->getMessage());
        }
    }
    
    /**
     * 5 dakika içindeki randevuları kontrol et
     */
    private function checkUpcomingAppointments()
    {
        try {
            $db = \Config\Database::connect();
            $notificationService = service('NotificationService');
            $userModel = model('App\Models\UserModel');
            
            // 5 dakika içindeki randevuları bul
            $builder = $db->table('enquire e');
            $builder->select('e.*, p.address as property_address');
            $builder->join('property p', 'e.property_id = p.id', 'left');
            $builder->where('e.status !=', 'closed');
            $builder->where('e.appointment_date', date('Y-m-d'));
            $builder->where('e.appointment_time IS NOT NULL');
            $builder->where('e.appointment_time !=', '');
            $builder->where('e.appointment_time !=', 'En yakın zaman');
            
            $enquiries = $builder->get()->getResultArray();
            
            $reminderCount = 0;
            
            foreach ($enquiries as $enquiry) {
                // Randevu saatini parse et
                $appointmentTimeStr = $enquiry['appointment_time'];
                
                if (strpos($appointmentTimeStr, '-') !== false) {
                    $timeParts = explode('-', $appointmentTimeStr);
                    $startTime = trim($timeParts[0]);
                    $appointmentDateTime = $enquiry['appointment_date'] . ' ' . $startTime . ':00';
                    
                    $appointmentTimestamp = strtotime($appointmentDateTime);
                    $now = time();
                    $timeDiff = $appointmentTimestamp - $now;
                    
                    // 4-6 dakika aralığında ise bildirim gönder
                    if ($timeDiff >= 240 && $timeDiff <= 360) {
                        
                        // Son 10 dakika içinde bu randevu için bildirim gönderilmiş mi kontrol et
                        $builder = $db->table('user_activity_log');
                        $builder->where('activity_type', 'appointment_reminder_5min');
                        $builder->where('property_id', $enquiry['property_id']);
                        $builder->where('created_at >', date('Y-m-d H:i:s', strtotime('-10 minutes')));
                        $recentReminder = $builder->get()->getRowArray();
                        
                        if (!$recentReminder) {
                            // İlan bilgilerini al
                            $propertyModel = model('App\Models\ListingModel');
                            $property = $propertyModel->where('property.id', $enquiry['property_id'])
                                                      ->where('language_id', getenv('LANGUAGE_ID'))
                                                      ->findAll(1);
                            
                            $propertyTitle = 'Mülk ID: ' . $enquiry['property_id'];
                            if ($property && count($property) > 0) {
                                $propData = json_decode($property[0]['json_object'], true);
                                if ($propData && isset($propData['field_10'])) {
                                    $propertyTitle = $propData['field_10'];
                                }
                            }
                            
                            // Tüm adminlere bildirim gönder
                            $admins = $userModel->where('type', 'ADMIN')->findAll();
                            
                            foreach ($admins as $admin) {
                                $title = '🔔 Randevu Hatırlatması (5 dk)';
                                $body = $enquiry['name_surname'] . ' müşterisi 5 dakika içinde aranacak - ' . $propertyTitle;
                                
                                $notificationData = [
                                    'property_id' => $enquiry['property_id'],
                                    'enquiry_id' => $enquiry['id'],
                                    'click_action' => 'OPEN_ENQUIRY_DETAIL',
                                    'notification_type' => 'appointment_reminder_5min',
                                    'customer_name' => $enquiry['name_surname'],
                                    'customer_phone' => $enquiry['phone'],
                                    'appointment_time' => $appointmentDateTime
                                ];
                                
                                // Bildirim gönder
                                $notificationService->sendToUser(
                                    $admin['id'],
                                    $title,
                                    $body,
                                    $notificationData
                                );
                            }
                            
                            // Activity log'a kaydet (tekrar gönderimi önlemek için)
                            $builder = $db->table('user_activity_log');
                            $builder->insert([
                                'user_id' => null,
                                'activity_type' => 'appointment_reminder_5min',
                                'property_id' => $enquiry['property_id'],
                                'activity_data' => json_encode(['enquiry_id' => $enquiry['id']]),
                                'created_at' => date('Y-m-d H:i:s')
                            ]);
                            
                            $reminderCount++;
                            
                            CLI::write("⏰ 5 dk hatırlatması gönderildi: {$enquiry['name_surname']} - $propertyTitle", 'green');
                        }
                    }
                }
            }
            
            if ($reminderCount > 0) {
                CLI::write("✅ $reminderCount adet 5 dakika hatırlatıcısı gönderildi", 'green');
            }
            
        } catch (\Exception $e) {
            CLI::write('❌ 5 dakika hatırlatıcısı hatası: ' . $e->getMessage(), 'red');
            log_message('error', '5 minute reminder error: ' . $e->getMessage());
        }
    }
}

/**
 * KURULUM TALİMATLARI:
 * 
 * 1. Bu dosyayı app/Commands/AppointmentReminder.php olarak kaydedin
 * 
 * 2. Veritabanına appointment_reminders tablosunu ekleyin:
 * 
 * CREATE TABLE `appointment_reminders` (
 *   `id` int(11) NOT NULL AUTO_INCREMENT,
 *   `enquiry_id` int(11) NOT NULL,
 *   `reminder_time` datetime NOT NULL,
 *   `appointment_time` datetime NOT NULL,
 *   `status` enum('pending','sent','cancelled') DEFAULT 'pending',
 *   `sent_at` datetime DEFAULT NULL,
 *   `created_at` datetime DEFAULT CURRENT_TIMESTAMP,
 *   PRIMARY KEY (`id`),
 *   KEY `idx_reminder_time` (`reminder_time`),
 *   KEY `idx_enquiry_id` (`enquiry_id`)
 * ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
 * 
 * 3. Crontab'a ekleyin (her dakika çalışacak):
 * * * * * * cd /path/to/project && php spark appointments:remind >> /path/to/logs/appointments.log 2>&1
 * 
 * 4. Test etmek için manuel çalıştırın:
 * php spark appointments:remind
 */