<?php

namespace App\Controllers;
use CodeIgniter\RESTful\ResourceController;
use CodeIgniter\API\ResponseTrait;

class DeviceController extends ResourceController
{

  use ResponseTrait;
  
   /**
     * Gerçek client IP adresini al - CloudFlare, proxy, load balancer desteği ile
     */
    private function getRealClientIP()
    {
        $ip_keys = array(
            'HTTP_CF_CONNECTING_IP',      // CloudFlare
            'HTTP_X_REAL_IP',              // Nginx proxy
            'HTTP_CLIENT_IP',              // Shared Internet
            'HTTP_X_FORWARDED_FOR',        // Proxy
            'HTTP_X_FORWARDED',
            'HTTP_X_CLUSTER_CLIENT_IP',
            'HTTP_FORWARDED_FOR',
            'HTTP_FORWARDED',
            'REMOTE_ADDR'                  // Direct connection
        );
        
        foreach ($ip_keys as $key) {
            if (array_key_exists($key, $_SERVER) === true) {
                foreach (explode(',', $_SERVER[$key]) as $ip) {
                    $ip = trim($ip);
                    
                    // Özel IP aralıklarını ve reserved IP'leri filtrele
                    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false) {
                        return $ip;
                    }
                }
            }
        }
        
        // Fallback - en son çare olarak REMOTE_ADDR
        return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    }
  
    public function update_device_info()
    {
        try {
            $data_json = $this->request->getJSON();
            $deviceModel = model('App\Models\DeviceModel');
            
            // Zorunlu parametre kontrolü
            if (empty($data_json->device_uuid)) {
                return $this->respond([
                    'code'     => 'failed',
                    'message'  => 'Device UUID eksik'
                ]);
            }
            
            $deviceUuid = $data_json->device_uuid;
            
            // Request verilerini hazırla
            $deviceData = [
                 'device_uuid' => $deviceUuid, // ✅ EKLE
                'last_active' => date('Y-m-d H:i:s'),
                'is_active'   => 1
             
            ];
            
            // Kullanıcı bilgilerini ekle
            if (!empty($data_json->user_id)) {
                $deviceData['user_id'] = $data_json->user_id;
            }
            
            if (!empty($data_json->email)) {
                $deviceData['email'] = $data_json->email;
            }
            
            // FCM Token bilgilerini ekle
            if (!empty($data_json->fcmToken)) {
                $deviceData['fcmToken'] = $data_json->fcmToken;
                $deviceData['push_token'] = $data_json->fcmToken; // Geriye dönük uyumluluk
            }
            
            // Platform bilgileri
            if (!empty($data_json->platform)) {
                $deviceData['platform'] = $data_json->platform;
            }
            
            // Cihaz bilgileri
            if (!empty($data_json->device_name)) {
                $deviceData['device_name'] = $data_json->device_name;
            }
            
            if (!empty($data_json->system_name)) {
                $deviceData['system_name'] = $data_json->system_name;
            }
            
            if (!empty($data_json->system_version)) {
                $deviceData['system_version'] = $data_json->system_version;
            }
            
            if (!empty($data_json->brand)) {
                $deviceData['brand'] = $data_json->brand;
            }
            
            if (!empty($data_json->device_brand)) {
                $deviceData['device_brand'] = $data_json->device_brand;
            }
            
            if (!empty($data_json->device_model)) {
                $deviceData['device_model'] = $data_json->device_model;
            }
            
            if (!empty($data_json->manufacturer)) {
                $deviceData['manufacturer'] = $data_json->manufacturer;
            }
            
            if (!empty($data_json->android_version)) {
                $deviceData['android_version'] = $data_json->android_version;
            }
            
            if (!empty($data_json->app_version)) {
                $deviceData['app_version'] = $data_json->app_version;
            }
            
            if (!empty($data_json->app_build_number)) {
                $deviceData['app_build_number'] = $data_json->app_build_number;
            }
            
            // Konum bilgileri
            if (!empty($data_json->country_code)) {
                $deviceData['country_code'] = $data_json->country_code;
            }
            
            if (!empty($data_json->country)) {
                $deviceData['country'] = $data_json->country;
            }
            
            if (!empty($data_json->region)) {
                $deviceData['region'] = $data_json->region;
            }
            
            if (!empty($data_json->language)) {
                $deviceData['language'] = $data_json->language;
            }
            
            if (!empty($data_json->ip_address)) {
                $deviceData['ip_address'] = $data_json->ip_address;
            }
            
            // İzin bilgileri
            if (isset($data_json->push_consent)) {
                $deviceData['push_consent'] = (bool)$data_json->push_consent ? 1 : 0;
            }
            
            if (isset($data_json->marketing_consent)) {
                $deviceData['marketing_consent'] = (bool)$data_json->marketing_consent ? 1 : 0;
            }
            
            // İlk görülme tarihi (sadece yeni cihazlar için)
            $deviceData['first_seen'] = date('Y-m-d H:i:s');
            
            // Topic abonelikleri için
            $subscribeTopics = [];
            $unsubscribeTopics = [];
            
            // İzin durumuna göre topic abonelikleri güncelle
            if (isset($data_json->push_consent) && (bool)$data_json->push_consent) {
                if (!empty($data_json->user_id)) {
                    $subscribeTopics[] = 'user_' . $data_json->user_id;
                    $unsubscribeTopics[] = 'anonymous'; 
                } else {
                    $subscribeTopics[] = 'anonymous';
                }
                
                if (isset($data_json->marketing_consent) && (bool)$data_json->marketing_consent) {
                    $subscribeTopics[] = 'marketing';
                } else {
                    $unsubscribeTopics[] = 'marketing';
                }
            } else if (isset($data_json->push_consent) && !(bool)$data_json->push_consent) {
                // Push izni kaldırıldı, tüm topic'lerden çıkar
                $unsubscribeTopics = ['anonymous', 'marketing'];
                if (!empty($data_json->user_id)) {
                    $unsubscribeTopics[] = 'user_' . $data_json->user_id;
                }
            }
            
            // Mevcut cihazı kontrol et
            $existingDevice = $deviceModel->where('device_uuid', $deviceUuid)->first();
            
            // Cihaz UUID değişmiş olabilir ya da boş olabilir - kullanıcı kontrolü yap
            if (!$existingDevice && !empty($data_json->user_id) && !empty($data_json->fcmToken)) {
                // User ID ve FCM token ile cihaz ara
                $devicesByToken = $deviceModel->where('fcmToken', $data_json->fcmToken)
                                              ->where('user_id', $data_json->user_id)
                                              ->orderBy('last_active', 'DESC')
                                              ->findAll();
                                              
                // Aynı FCM token ile cihaz varsa, en son kullanılanı al
                if (count($devicesByToken) > 0) {
                    $existingDevice = $devicesByToken[0];
                } else {
                    // FCM token yoksa user_id ile son cihazı bul
                    $devicesByUser = $deviceModel->where('user_id', $data_json->user_id)
                                                ->orderBy('last_active', 'DESC')
                                                ->findAll();
                                                
                    if (count($devicesByUser) > 0) {
                        $existingDevice = $devicesByUser[0];
                    }
                }
            }
            
            if ($existingDevice) {
                // Mevcut cihazı güncelle
                try {
                    // First seen bilgisini korumak için
                    if (!empty($existingDevice['first_seen'])) {
                        unset($deviceData['first_seen']);
                    }
                    
                    // Mevcut topic abonelikleri
                    $currentTopics = !empty($existingDevice['topic_subscriptions']) ? 
                                     json_decode($existingDevice['topic_subscriptions'], true) : [];
                                     
                    // Yeni topic abonelikleri
                    $updatedTopics = $currentTopics;
                    
                    // Abone olunacak topic'leri ekle
                    foreach ($subscribeTopics as $topic) {
                        if (!in_array($topic, $updatedTopics)) {
                            $updatedTopics[] = $topic;
                        }
                    }
                    
                    // Abonelikten çıkılacak topic'leri çıkar
                    $updatedTopics = array_diff($updatedTopics, $unsubscribeTopics);
                    
                    // JSON formatında güncelle
                    $deviceData['topic_subscriptions'] = json_encode(array_values($updatedTopics));
                    
                    // Güncelleme yap
                    $deviceModel->update($existingDevice['id'], $deviceData);
                    
                    return $this->respond([
                        'code'      => 'success',
                        'message'   => 'Cihaz bilgileri başarıyla güncellendi',
                        
'device_id'  => $existingDevice['id'],
                        'updated'   => true
                    ]);
                } catch (\Exception $e) {
                    return $this->respond([
                        'code'     => 'failed',
                        'message'  => 'Cihaz güncellenirken hata oluştu: ' . $e->getMessage()
                    ]);
                }
            } else {
                // Yeni cihaz oluştur
                try {
                    // Yeni cihazda topic abonelikleri
                    $deviceData['topic_subscriptions'] = json_encode($subscribeTopics);
                    
                    // Aktivasyon sayısını ekle
                    $deviceData['activation_count'] = 1;
                    
                    // Cihaz uuid'yi ekle
                    $deviceData['device_uuid'] = $deviceUuid;
                    
                    // Yeni cihaz ekle
                    $deviceId = $deviceModel->insert($deviceData);
                    
                    return $this->respond([
                        'code'      => 'success',
                        'message'   => 'Yeni cihaz başarıyla eklendi',
                        'device_id' => $deviceId,
                        'created'   => true
                    ]);
                } catch (\Exception $e) {
                    return $this->respond([
                        'code'     => 'failed',
                        'message'  => 'Yeni cihaz eklenirken hata oluştu: ' . $e->getMessage()
                    ]);
                }
            }
        } catch (\Exception $e) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Cihaz bilgileri işlenirken hata oluştu: ' . $e->getMessage()
            ]);
        }
    }
                        
                        
                        
   /**
     * Cihazı kaydet veya güncelle
     */
    public function register_device()
    {
        $data = [
            'code'     => 'empty_code',
            'message'  => 'Empty Message'
        ];

        $data_json = $this->request->getJSON();
        $deviceModel = model('App\Models\DeviceModel');

        // Zorunlu parametreleri kontrol et
        if (empty($data_json->device_uuid)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Device UUID eksik'
            ]);
        }

        // İzin durumları
        $pushConsent = isset($data_json->push_consent) ? 
                      ($data_json->push_consent === true || $data_json->push_consent === '1' ? 1 : 0) : null;
        
        $marketingConsent = isset($data_json->marketing_consent) ? 
                           ($data_json->marketing_consent === true || $data_json->marketing_consent === '1' ? 1 : 0) : null;

        // Cihaz verileri hazırla
        $deviceData = [
            'device_uuid' => $data_json->device_uuid,
            'is_active' => 1,
            'last_active' => date('Y-m-d H:i:s')
        
        ];

        // FCM Token
        if (!empty($data_json->fcmToken)) {
            $deviceData['fcmToken'] = $data_json->fcmToken;
            $deviceData['push_token'] = $data_json->fcmToken; // Eski uyumluluk için
        }

        // Cihaz bilgileri
        if (!empty($data_json->platform)) {
            $deviceData['platform'] = $data_json->platform;
        }
        
        if (!empty($data_json->device_name)) {
            $deviceData['device_name'] = $data_json->device_name;
        }
        
        if (!empty($data_json->system_name)) {
            $deviceData['system_name'] = $data_json->system_name;
        }
        
        if (!empty($data_json->system_version)) {
            $deviceData['system_version'] = $data_json->system_version;
        }
        
        if (!empty($data_json->device_model)) {
            $deviceData['device_model'] = $data_json->device_model;
        }
        
        if (!empty($data_json->brand)) {
            $deviceData['brand'] = $data_json->brand;
        }
        
        if (!empty($data_json->device_brand)) {
            $deviceData['device_brand'] = $data_json->device_brand;
        }
        
        if (!empty($data_json->manufacturer)) {
            $deviceData['manufacturer'] = $data_json->manufacturer;
        }
        
        if (!empty($data_json->app_version)) {
            $deviceData['app_version'] = $data_json->app_version;
        }
        
        if (!empty($data_json->android_version)) {
            $deviceData['android_version'] = $data_json->android_version;
        }
        
        // Dil ve ülke bilgisi
        if (!empty($data_json->language)) {
            $deviceData['language'] = $data_json->language;
        }
        
        if (!empty($data_json->country_code)) {
            $deviceData['country_code'] = $data_json->country_code;
        }
        
        if (!empty($data_json->country)) {
            $deviceData['country'] = $data_json->country;
        }
        
        if (!empty($data_json->region)) {
            $deviceData['region'] = $data_json->region;
        }
        
        // IP adresi
        $deviceData['ip_address'] = $this->request->getIPAddress();
        
        // İzin durumları
        if ($pushConsent !== null) {
            $deviceData['push_consent'] = $pushConsent;
        }
        
        if ($marketingConsent !== null) {
            $deviceData['marketing_consent'] = $marketingConsent;
        }

        // Kullanıcı kimlik bilgileri varsa doğrula
        if (!empty($data_json->user_id) && !empty($data_json->token)) {
            $tokenModel = model('App\Models\TokenModel');
            $token = $data_json->token;
            $user_token = $tokenModel->where('token', $token)->findAll(1);

            if (count($user_token) > 0 && $user_token[0]['user_id'] == $data_json->user_id) {
                $deviceData['user_id'] = $data_json->user_id;
                
                if (!empty($data_json->user_email)) {
                    $deviceData['email'] = $data_json->user_email;
                }
            } else {
                // Token eşleşmezse, user_id kullanma
                unset($deviceData['user_id']);
            }
        } else if (!empty($data_json->user_email)) {
            // Token olmadan sadece e-posta varsa
            $deviceData['email'] = $data_json->user_email;
        }

        try {
            // Cihazı kaydet veya güncelle
            $result = $deviceModel->saveDevice($deviceData);

            if ($result) {
                // Konu aboneliklerini güncelle
                $this->_updateTopicSubscriptions($data_json->device_uuid, $deviceData);
                
                return $this->respond([
                    'code'     => 'success',
                    'message'  => 'Cihaz başarıyla kaydedildi',
                    'device_uuid' => $data_json->device_uuid
                ]);
            } else {
                return $this->respond([
                    'code'     => 'failed',
                    'message'  => 'Cihaz kaydedilirken hata oluştu'
                ]);
            }
        } catch (\Exception $e) {
            log_message('error', 'Cihaz kaydedilirken hata: ' . $e->getMessage());
            
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Cihaz kaydedilirken hata: ' . $e->getMessage()
            ]);
        }
    }
    
    /**
     * FCM Token güncelleme
     */
    public function update_push_token()
    {
        $data = [
            'code'     => 'empty_code',
            'message'  => 'Empty Message'
        ];

        $data_json = $this->request->getJSON();
        $deviceModel = model('App\Models\DeviceModel');

        // Zorunlu parametreleri kontrol et
        if (empty($data_json->device_uuid)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Device UUID eksik'
            ]);
        }

        if (empty($data_json->fcm_token) && empty($data_json->fcmToken)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'FCM token eksik'
            ]);
        }

        $fcmToken = !empty($data_json->fcmToken) ? $data_json->fcmToken : $data_json->fcm_token;
        $deviceUuid = $data_json->device_uuid;

        // Ek verileri hazırla
        $additionalData = [
            'last_active' => date('Y-m-d H:i:s')
        ];

        // Kullanıcı kimlik bilgileri varsa doğrula
        if (!empty($data_json->user_id) && !empty($data_json->token)) {
            $tokenModel = model('App\Models\TokenModel');
            $token = $data_json->token;
            $user_token = $tokenModel->where('token', $token)->findAll(1);

            if (count($user_token) > 0 && $user_token[0]['user_id'] == $data_json->user_id) {
                $additionalData['user_id'] = $data_json->user_id;
            }
        }

        // Platform bilgisi varsa ekle
        if (!empty($data_json->platform)) {
            $additionalData['platform'] = $data_json->platform;
        }
        
        // Cihaz bilgileri varsa ekle
        if (!empty($data_json->device_name)) {
            $additionalData['device_name'] = $data_json->device_name;
        }
        
        if (!empty($data_json->device_model)) {
            $additionalData['device_model'] = $data_json->device_model;
        }
        
        if (!empty($data_json->os_version)) {
            $additionalData['system_version'] = $data_json->os_version;
        }
        
          // Konum ve dil bilgileri
        if (!empty($data_json->language)) {
            $additionalData['language'] = $data_json->language;
        }
        
        if (!empty($data_json->country_code)) {
            $additionalData['country_code'] = $data_json->country_code;
        }
        
        if (!empty($data_json->country)) {
            $additionalData['country'] = $data_json->country;
        }
        
        if (!empty($data_json->region)) {
            $additionalData['region'] = $data_json->region;
        }

        // Push ve marketing izinleri varsa ekle
        if (isset($data_json->push_consent)) {
            $additionalData['push_consent'] = $data_json->push_consent === true || $data_json->push_consent === '1' ? 1 : 0;
        }
        
        if (isset($data_json->marketing_consent)) {
            $additionalData['marketing_consent'] = $data_json->marketing_consent === true || $data_json->marketing_consent === '1' ? 1 : 0;
        }
        
        // IP adresi ekle
        $additionalData['ip_address'] = $this->request->getIPAddress();

        try {
            // FCM Token'ı güncelle
            $success = $deviceModel->updateFcmToken($deviceUuid, $fcmToken, $additionalData);

            if ($success) {
                // Konu aboneliklerini güncelle (izinlere göre)
                $this->_updateTopicSubscriptions($deviceUuid, $additionalData);
                
                return $this->respond([
                    'code'     => 'success',
                    'message'  => 'FCM token başarıyla güncellendi'
                ]);
            } else {
                return $this->respond([
                    'code'     => 'failed',
                    'message'  => 'FCM token güncellenirken hata oluştu'
                ]);
            }
        } catch (\Exception $e) {
            log_message('error', 'FCM token güncellenirken hata: ' . $e->getMessage());
            
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'FCM token güncellenirken hata: ' . $e->getMessage()
            ]);
        }
    }
    
    /**
     * Konu aboneliği güncelleme
     */
    public function update_topic_subscriptions()
    {
        $data = [
            'code'     => 'empty_code',
            'message'  => 'Empty Message'
        ];

        $data_json = $this->request->getJSON();
        $deviceModel = model('App\Models\DeviceModel');

        // Zorunlu parametreleri kontrol et
        if (empty($data_json->device_uuid)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Device UUID eksik'
            ]);
        }

        // Eklenecek ve çıkarılacak konuları kontrol et
        $topicsToAdd = !empty($data_json->subscribe_topics) && is_array($data_json->subscribe_topics) 
                     ? $data_json->subscribe_topics 
                     : [];
                     
        $topicsToRemove = !empty($data_json->unsubscribe_topics) && is_array($data_json->unsubscribe_topics) 
                        ? $data_json->unsubscribe_topics 
                        : [];
        
        if (empty($topicsToAdd) && empty($topicsToRemove)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Eklenecek veya çıkarılacak konu belirtilmedi'
            ]);
        }

        // Kullanıcı kimlik bilgileri varsa doğrula
        $userId = null;
        $isAuthenticated = false;
        
        if (!empty($data_json->user_id) && !empty($data_json->token)) {
            $tokenModel = model('App\Models\TokenModel');
            $token = $data_json->token;
            $user_token = $tokenModel->where('token', $token)->findAll(1);

            if (count($user_token) > 0 && $user_token[0]['user_id'] == $data_json->user_id) {
                $userId = $data_json->user_id;
                $isAuthenticated = true;
                
                // Güvenlik kontrolü - kullanıcı sadece kendi cihazlarını güncelleme hakkına sahip
                $device = $deviceModel->getDeviceByUuid($data_json->device_uuid);
                if ($device && !empty($device['user_id']) && $device['user_id'] != $userId) {
                    return $this->respond([
                        'code'     => 'failed',
                        'message'  => 'Bu cihazın konularını güncelleme yetkiniz yok'
                    ]);
                }
            }
        }

        try {
            $success = true;
            
            // Konuları ekle
            foreach ($topicsToAdd as $topic) {
                $result = $deviceModel->updateTopicSubscription($data_json->device_uuid, $topic, true);
                if (!$result) {
                    $success = false;
                }
            }
            
            // Konulardan çıkar
            foreach ($topicsToRemove as $topic) {
                $result = $deviceModel->updateTopicSubscription($data_json->device_uuid, $topic, false);
                if (!$result) {
                    $success = false;
                }
            }
            
            if ($success) {
                return $this->respond([
                    'code'     => 'success',
                    'message'  => 'Konu abonelikleri başarıyla güncellendi',
                    'device_uuid' => $data_json->device_uuid,
                    'subscribed' => $topicsToAdd,
                    'unsubscribed' => $topicsToRemove
                ]);
            } else {
                return $this->respond([
                    'code'     => 'failed',
                    'message'  => 'Konu abonelikleri güncellenirken hata oluştu'
                ]);
            }
        } catch (\Exception $e) {
            log_message('error', 'Konu abonelikleri güncellenirken hata: ' . $e->getMessage());
            
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Konu abonelikleri güncellenirken hata: ' . $e->getMessage()
            ]);
        }
    }
    
    /**
     * Cihaz durumunu güncelle (aktif/pasif)
     */
    public function update_device_status()
    {
        $data = [
            'code'     => 'empty_code',
            'message'  => 'Empty Message'
        ];

        $data_json = $this->request->getJSON();
        $deviceModel = model('App\Models\DeviceModel');

        // Zorunlu parametreleri kontrol et
        if (empty($data_json->device_uuid)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Device UUID eksik'
            ]);
        }

        if (!isset($data_json->is_active)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Cihaz durumu (is_active) eksik'
            ]);
        }

        $deviceUuid = $data_json->device_uuid;
        $isActive = $data_json->is_active === true || $data_json->is_active === '1' || $data_json->is_active === 1;

        // Kullanıcı kimlik bilgileri varsa doğrula
        $userId = null;
        if (!empty($data_json->user_id) && !empty($data_json->token)) {
            $tokenModel = model('App\Models\TokenModel');
            $token = $data_json->token;
            $user_token = $tokenModel->where('token', $token)->findAll(1);

            if (count($user_token) > 0 && $user_token[0]['user_id'] == $data_json->user_id) {
                $userId = $data_json->user_id;
                
                // Güvenlik kontrolü - kullanıcı sadece kendi cihazlarını güncelleme hakkına sahip
                $device = $deviceModel->getDeviceByUuid($deviceUuid);
                if ($device && !empty($device['user_id']) && $device['user_id'] != $userId) {
                    return $this->respond([
                        'code'     => 'failed',
                        'message'  => 'Bu cihazı güncelleme yetkiniz yok'
                    ]);
                }
            }
        }

        try {
            // Cihaz durumunu güncelle
            $success = $deviceModel->setDeviceActive($deviceUuid, $isActive);

            if ($success) {
                return $this->respond([
                    'code'     => 'success',
                    'message'  => 'Cihaz durumu başarıyla güncellendi',
                    'device_uuid' => $deviceUuid,
                    'is_active' => $isActive
                ]);
            } else {
                return $this->respond([
                    'code'     => 'failed',
                    'message'  => 'Cihaz durumu güncellenirken hata oluştu'
                ]);
            }
        } catch (\Exception $e) {
            log_message('error', 'Cihaz durumu güncellenirken hata: ' . $e->getMessage());
            
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Cihaz durumu güncellenirken hata: ' . $e->getMessage()
            ]);
        }
    }
    
    /**
     * Konu aboneliklerini güncelle - bildirim ve pazarlama izinlerine göre
     */
    private function _updateTopicSubscriptions($deviceUuid, $deviceData)
    {
        $deviceModel = model('App\Models\DeviceModel');
        $topicsToAdd = [];
        $topicsToRemove = [];
        
        // Bildirim izni kontrolü
        $hasPushConsent = isset($deviceData['push_consent']) && $deviceData['push_consent'] == 1;
        
        if ($hasPushConsent) {
            // Kullanıcı giriş yapmış mı?
            if (!empty($deviceData['user_id'])) {
                $topicsToAdd[] = 'user_' . $deviceData['user_id'];
                $topicsToRemove[] = 'anonymous';
            } else {
                $topicsToAdd[] = 'anonymous';
            }
            
            // Pazarlama izni var mı?
            $hasMarketingConsent = isset($deviceData['marketing_consent']) && $deviceData['marketing_consent'] == 1;
            
            if ($hasMarketingConsent) {
                $topicsToAdd[] = 'marketing';
            } else {
                $topicsToRemove[] = 'marketing';
            }
        } else {
            // Push izni yoksa hiçbir konuya abone olmamalı
            if (!empty($deviceData['user_id'])) {
                $topicsToRemove[] = 'user_' . $deviceData['user_id'];
            }
            $topicsToRemove[] = 'anonymous';
            $topicsToRemove[] = 'marketing';
        }
        
        // Konu aboneliklerini güncelle
        foreach ($topicsToAdd as $topic) {
            $deviceModel->updateTopicSubscription($deviceUuid, $topic, true);
        }
        
        foreach ($topicsToRemove as $topic) {
            $deviceModel->updateTopicSubscription($deviceUuid, $topic, false);
        }
    }
            
            

    /**
     * Konu aboneliklerini güncelle - bildirim ve pazarlama izinlerine göre
    
    private function _updateTopicSubscriptions($deviceUuid, $deviceData)
    {
        $deviceModel = model('App\Models\DeviceModel');
        $topicsToAdd = [];
        $topicsToRemove = [];
        
        // Bildirim izni kontrolü
        $hasPushConsent = isset($deviceData['push_consent']) && $deviceData['push_consent'] == 1;
        
        if ($hasPushConsent) {
            // Kullanıcı giriş yapmış mı?
            if (!empty($deviceData['user_id'])) {
                $topicsToAdd[] = 'user_' . $deviceData['user_id'];
                $topicsToRemove[] = 'anonymous';
            } else {
                $topicsToAdd[] = 'anonymous';
            }
            
            // Pazarlama izni var mı?
            $hasMarketingConsent = isset($deviceData['marketing_consent']) && $deviceData['marketing_consent'] == 1;
            
            if ($hasMarketingConsent) {
                $topicsToAdd[] = 'marketing';
            } else {
                $topicsToRemove[] = 'marketing';
            }
        } else {
            // Push izni yoksa hiçbir konuya abone olmamalı
            if (!empty($deviceData['user_id'])) {
                $topicsToRemove[] = 'user_' . $deviceData['user_id'];
            }
            $topicsToRemove[] = 'anonymous';
            $topicsToRemove[] = 'marketing';
        }
        
        // Konu aboneliklerini güncelle
        foreach ($topicsToAdd as $topic) {
            $deviceModel->updateTopicSubscription($deviceUuid, $topic, true);
        }
        
        foreach ($topicsToRemove as $topic) {
            $deviceModel->updateTopicSubscription($deviceUuid, $topic, false);
        }
    } */
    
    /**
     * Kullanıcının cihazlarını getir
     */
    public function get_user_devices()
    {
        // Token kontrolü
        if (empty($_GET['token'])) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Token eksik'
            ]);
        }

        $token = $_GET['token'];
        $tokenModel = model('App\Models\TokenModel');
        
        $user_token = $tokenModel->where('token', $token)->findAll(1);

        if (count($user_token) == 0) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Geçerli token eksik'
            ]);
        }
        
        $userId = $user_token[0]['user_id'];
        
        if (empty($_GET['user_id'])) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Kullanıcı ID eksik'
            ]);
        }

        if ($userId != $_GET['user_id']) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Token yetkileri eksik'
            ]);
        }
        
        // Tüm cihazları dahil et (pasif olanları da)
        $includeInactive = isset($_GET['include_inactive']) && 
                           ($_GET['include_inactive'] === '1' || $_GET['include_inactive'] === 'true');
        
        // DeviceModel yükle
        $deviceModel = model('App\Models\DeviceModel');
        
        // Kullanıcının cihazlarını getir
        $devices = $deviceModel->getUserDevices($userId, !$includeInactive);
        
        // Cihaz sayısı
        $totalCount = count($devices);
        
        // Her cihaz için konu aboneliklerini al
        foreach ($devices as &$device) {
            $topics = $deviceModel->getDeviceTopics($device['device_uuid']);
            $device['topics'] = $topics;
            
            // Hassas bilgileri kaldır
            unset($device['push_token']);
            
         // Aktif olma zamanını uygun formatta göster
            if (!empty($device['last_active'])) {
                $device['last_active_formatted'] = date('d.m.Y H:i', strtotime($device['last_active']));
            }
            
            if (!empty($device['first_seen'])) {
                $device['first_seen_formatted'] = date('d.m.Y H:i', strtotime($device['first_seen']));
            }
            
            // Platform ve model bilgilerini tek alanda birleştir
            $device['device_info'] = ($device['platform'] ?? '') . ' ' . 
                                    ($device['device_model'] ?? '') . ' ' . 
                                    ($device['system_version'] ?? '');
            $device['device_info'] = trim($device['device_info']);
        }
        
        return $this->respond([
            'code'     => 'success',
            'devices'  => $devices,
            'total'    => $totalCount
        ]);
    }
    
    /**
     * Cihaz sil veya kullanıcıdan ayır
     */
    public function remove_device()
    {
        $data = [
            'code'     => 'empty_code',
            'message'  => 'Empty Message'
        ];

        $data_json = $this->request->getJSON();
        $deviceModel = model('App\Models\DeviceModel');

        // Zorunlu parametreleri kontrol et
        if (empty($data_json->device_uuid)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Device UUID eksik'
            ]);
        }

        // Token kontrolü
        if (empty($data_json->token)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Token eksik'
            ]);
        }

        $token = $data_json->token;
        $tokenModel = model('App\Models\TokenModel');
        
        $user_token = $tokenModel->where('token', $token)->findAll(1);

        if (count($user_token) == 0) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Geçerli token eksik'
            ]);
        }
        
        $userId = $user_token[0]['user_id'];
        
        // Kullanıcının cihaz üzerinde yetkisi var mı kontrol et
        $device = $deviceModel->getDeviceByUuid($data_json->device_uuid);
        
        if (!$device) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Cihaz bulunamadı'
            ]);
        }
        
        if ($device['user_id'] != $userId) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Bu cihazı silme/ayırma yetkiniz yok'
            ]);
        }
        
        // İşlem tipini kontrol et
        $actionType = isset($data_json->action_type) ? $data_json->action_type : 'detach';
        
        try {
            if ($actionType === 'delete') {
                // Cihazı tamamen sil
                $success = $deviceModel->delete($device['id']);
                
                // Konu aboneliklerini de sil
                $db = \Config\Database::connect();
                $db->table('device_topic_subscriptions')
                   ->where('device_uuid', $data_json->device_uuid)
                   ->delete();
                
                $message = 'Cihaz başarıyla silindi';
            } else {
                // Cihazı kullanıcıdan ayır (user_id'yi NULL yap ve pasifleştir)
                $updateData = [
                    'user_id' => null,
                    'push_consent' => 0,
                    'marketing_consent' => 0,
                    'is_active' => 0
                ];
                
                $success = $deviceModel->update($device['id'], $updateData);
                
                // Kullanıcıya özel konulardan çıkar
                $deviceModel->updateTopicSubscription($data_json->device_uuid, 'user_' . $userId, false);
                $deviceModel->updateTopicSubscription($data_json->device_uuid, 'marketing', false);
                
                $message = 'Cihaz başarıyla kullanıcıdan ayrıldı';
            }
            
            if ($success) {
                return $this->respond([
                    'code'     => 'success',
                    'message'  => $message,
                    'device_uuid' => $data_json->device_uuid
                ]);
            } else {
                return $this->respond([
                    'code'     => 'failed',
                    'message'  => 'İşlem sırasında hata oluştu'
                ]);
            }
        } catch (\Exception $e) {
            log_message('error', 'Cihaz silme/ayırma işlemi sırasında hata: ' . $e->getMessage());
            
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'İşlem sırasında hata: ' . $e->getMessage()
            ]);
        }
    }
    

    /**
     * Cihaz bilgilerini getir
     */
    public function get_device_info()
    {
        // Cihaz UUID'si gerekli
        if (empty($_GET['device_uuid'])) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Device UUID eksik'
            ]);
        }

        $deviceUuid = $_GET['device_uuid'];
        $deviceModel = model('App\Models\DeviceModel');
        
        // Kullanıcı kimlik doğrulama bilgileri (opsiyonel)
        $userId = null;
        $isAuthenticated = false;
        
        if (!empty($_GET['token']) && !empty($_GET['user_id'])) {
            $tokenModel = model('App\Models\TokenModel');
            $token = $_GET['token'];
            $user_token = $tokenModel->where('token', $token)->findAll(1);

            if (count($user_token) > 0 && $user_token[0]['user_id'] == $_GET['user_id']) {
                $userId = $_GET['user_id'];
                $isAuthenticated = true;
            }
        }
        
        // Cihaz bilgilerini getir
        $device = $deviceModel->getDeviceByUuid($deviceUuid);
        
        if (!$device) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Cihaz bulunamadı'
            ]);
        }
        
        // Güvenlik kontrolü - kullanıcı kimliği doğrulanmışsa, cihaz o kullanıcıya ait olmalı
        if ($isAuthenticated && !empty($device['user_id']) && $device['user_id'] != $userId) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Bu cihazın bilgilerini görüntüleme yetkiniz yok'
            ]);
        }
        
        // Hassas bilgileri kaldır
        unset($device['push_token']);
        
        // Zaman damgaları için formatlanmış versiyonlar ekle
        if (!empty($device['last_active'])) {
            $device['last_active_formatted'] = date('d.m.Y H:i', strtotime($device['last_active']));
        }
        
        if (!empty($device['first_seen'])) {
            $device['first_seen_formatted'] = date('d.m.Y H:i', strtotime($device['first_seen']));
        }
        
        // Konu aboneliklerini ekle
        $topics = $deviceModel->getDeviceTopics($deviceUuid);
        $device['topics'] = $topics;
        
        // Consent bilgilerini ekle
        $consentModel = model('App\Models\ConsentModel');
        $consents = $consentModel->getConsentsByDeviceUuid($deviceUuid);
        
        if (!empty($consents)) {
            $latestConsent = $consents[0];
            
            $device['consents'] = [
                'kvkk_consent' => $latestConsent['kvkk_consent'] === '1',
                'gdpr_consent' => $latestConsent['gdpr_consent'] === '1',
                'privacy_consent' => $latestConsent['privacy_consent'] === '1',
                'marketing_consent' => $latestConsent['marketing_consent'] === '1',
                'analytics_consent' => $latestConsent['analytics_consent'] === '1',
                'device_info_consent' => $latestConsent['device_info_consent'] === '1',
                'push_consent' => $latestConsent['push_consent'] === '1',
                'consent_version' => $latestConsent['consent_version'] ?? '',
                'consent_timestamp' => $latestConsent['created_at'] ?? ''
            ];
        } else {
            $device['consents'] = null;
        }
        
        return $this->respond([
            'code'     => 'success',
            'device'   => $device
        ]);
    }
    
 
    
    
}