<?php

namespace App\Controllers;

use CodeIgniter\RESTful\ResourceController;
use CodeIgniter\API\ResponseTrait;

class NotificationController extends ResourceController
{
    use ResponseTrait;
    
    
     /**
     * Bildirimi alındı olarak işaretle
     */
    public function mark_delivered()
    {
        $data = [
            'code'     => 'empty_code',
            'message'  => 'Empty Message'
        ];

        $data_json = $this->request->getJSON();
        
        // Token kontrolü
        if (empty($data_json->token)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Token eksik'
            ]);
        }

        $token = $data_json->token;
        $tokenModel = model('App\Models\TokenModel');
        
        $user_token = $tokenModel->where('token', $token)->findAll(1);

        if (count($user_token) == 0) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Geçerli token eksik'
            ]);
        }
        
        $userId = $user_token[0]['user_id'];
        
        // Bildirim ID kontrolü
        if (empty($data_json->notification_id)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Bildirim ID eksik'
            ]);
        }
        
        $notificationId = (int)$data_json->notification_id;
        
        // NotificationModel yükle
        $notificationModel = model('App\Models\NotificationModel');
        
        // Bildirimi kontrol et
        $notification = $notificationModel->find($notificationId);
        
        if (!$notification) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Bildirim bulunamadı'
            ]);
        }
        
        // Yetki kontrolü
        if ($notification['user_id'] != $userId) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Bu bildirimi işaretleme yetkiniz yok'
            ]);
        }
        
        // Bildirimi alındı olarak işaretle
        $success = $notificationModel->update($notificationId, [
            'status' => 'delivered',
        ]);
        
        if ($success) {
            return $this->respond([
                'code'     => 'success',
                'message'  => 'Bildirim alındı olarak işaretlendi',
                'notification_id' => $notificationId
            ]);
        } else {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Bildirim işaretlenirken hata oluştu'
            ]);
        }
    }
    
    /**
 * Soruya cevap eklendiğinde bildirim gönder
 * 
 * @param int $enquiryId Enquiry ID
 * @param int $replyId Cevap ID
 * @param string $replierName Cevap yazan kişi
 * @param string $propertyTitle Mülk başlığı
 * @param string $message Cevap içeriği
 * @param bool $isAgent Cevap veren bir temsilci mi
 * @param int|null $agentId Temsilci ID
 * @return bool Başarı durumu
 */
public function sendEnquiryReplyNotification($enquiryId, $replyId, $replierName, $propertyTitle, $message, $isAgent = false, $agentId = null)
{
    // Özet mesaj (max 100 karakter)
    $shortMessage = mb_strlen($message) > 100 
        ? mb_substr($message, 0, 97) . '...' 
        : $message;
    
    // Bildirim içeriği
    $title = 'Yeni Cevap: ' . $propertyTitle;
    $body = $replierName . ': ' . $shortMessage;
    
    // Ek veriler
    $data = [
        'enquiry_id' => $enquiryId,
        'reply_id' => $replyId,
        'notification_type' => 'enquiry_reply',
        'click_action' => 'OPEN_ENQUIRY_DETAIL',
        'property_title' => $propertyTitle
    ];
    
    // Bildirim modeli
    $notificationModel = model('App\Models\NotificationModel');
    
    if ($isAgent) {
        // Temsilci cevapladıysa, adminlere bildirim gönder
        return $notificationModel->sendEnquiryNotification(
            $enquiryId,
            $title,
            $body,
            $data
        );
    } else {
        // Admin cevapladıysa, temsilciye bildirim gönder
        if ($agentId) {
            return $notificationModel->sendEnquiryNotification(
                $enquiryId,
                $title,
                $body,
                $data,
                $agentId
            );
        }
    }
    
    return true;
}

/**
 * EnquiryService'den gelen bildirimleri gönderir
 * 
 * @param int $enquiryId Enquiry ID
 * @param string $title Bildirim başlığı
 * @param string $body Bildirim içeriği
 * @param array $data Bildirim ek bilgileri
 * @param int|null $userId Hedef kullanıcı ID (yoksa tüm adminlere gönderilir)
 * @return bool Başarı durumu
 */
public function sendEnquiryNotification($enquiryId, $title, $body, $data = [], $userId = null)
{
    $notificationService = service('NotificationService');
    
    // Ek verileri düzenle
    $data['enquiry_id'] = $enquiryId;
    $data['notification_type'] = 'enquiry';
    $data['click_action'] = 'OPEN_ENQUIRY_DETAIL';
    
    // Bildirim modeli
    $notificationModel = model('App\Models\NotificationModel');
    
    // Bildirim kimliği oluştur
    $notificationId = $notificationModel->addNotification([
        'user_id' => $userId,
        'title' => $title,
        'body' => $body,
        'data' => json_encode($data),
        'target' => 'enquiry',
        'status' => 'sent'
    ]);
    
    if ($userId) {
        // Belirli bir kullanıcıya gönder
        return $notificationService->sendToUser(
            $userId,
            $title,
            $body,
            $data
        );
    } else {
        // Tüm adminlere gönder
        $userModel = model('App\Models\UserModel');
        $admins = $userModel->where('type', 'ADMIN')->findAll();
        
        $success = true;
        foreach ($admins as $admin) {
            $result = $notificationService->sendToUser(
                $admin['id'],
                $title,
                $body,
                $data
            );
            
            if (!$result) {
                $success = false;
            }
        }
        
        return $success;
    }
}
    
    /**
     * Bildirimi okundu olarak işaretle
     */
    public function mark_read()
    {
        $data = [
            'code'     => 'empty_code',
            'message'  => 'Empty Message'
        ];

        $data_json = $this->request->getJSON();
        
        // Token kontrolü
        if (empty($data_json->token)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Token eksik'
            ]);
        }

        $token = $data_json->token;
        $tokenModel = model('App\Models\TokenModel');
        
        $user_token = $tokenModel->where('token', $token)->findAll(1);

        if (count($user_token) == 0) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Geçerli token eksik'
            ]);
        }
        
        $userId = $user_token[0]['user_id'];
        
        // Bildirim ID kontrolü
        if (empty($data_json->notification_id)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Bildirim ID eksik'
            ]);
        }
        
        $notificationId = (int)$data_json->notification_id;
        
        // NotificationModel yükle
        $notificationModel = model('App\Models\NotificationModel');
        
        // Bildirimi kontrol et
        $notification = $notificationModel->find($notificationId);
        
        if (!$notification) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Bildirim bulunamadı'
            ]);
        }
        
        // Yetki kontrolü
        if ($notification['user_id'] != $userId) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Bu bildirimi işaretleme yetkiniz yok'
            ]);
        }
        
        // Bildirimi okundu olarak işaretle
        $success = $notificationModel->markAsRead($notificationId);
        
        if ($success) {
            return $this->respond([
                'code'     => 'success',
                'message'  => 'Bildirim okundu olarak işaretlendi',
                'notification_id' => $notificationId
            ]);
        } else {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Bildirim işaretlenirken hata oluştu'
            ]);
        }
    }
    
    /**
     * Kullanıcının bildirimlerini getir
     */
    public function getUserNotifications()
    {
        // Token kontrolü
        if (empty($_GET['token'])) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Token eksik'
            ]);
        }

        $token = $_GET['token'];
        $tokenModel = model('App\Models\TokenModel');
        
        $user_token = $tokenModel->where('token', $token)->findAll(1);

        if (count($user_token) == 0) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Geçerli token eksik'
            ]);
        }
        
        $userId = $user_token[0]['user_id'];
        
        // Sorgu parametreleri
        $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 50;
        $offset = isset($_GET['offset']) ? (int)$_GET['offset'] : 0;
        $targetType = isset($_GET['target_type']) ? $_GET['target_type'] : null;
        
        // NotificationModel yükle
        $notificationModel = model('App\Models\NotificationModel');
        
        // Bildirim geçmişini al
        $notifications = $notificationModel->getNotificationHistory($userId, $limit, $offset, $targetType);
        $totalCount = $notificationModel->getNotificationCount($userId, $targetType);
        $unreadCount = $notificationModel->getUnreadCount($userId);
        
        return $this->respond([
            'code'     => 'success',
            'notifications' => $notifications,
            'total' => $totalCount,
            'unread' => $unreadCount,
            'limit' => $limit,
            'offset' => $offset,
        ]);
    }
    
    /**
     * Bildirimi sil
     */
    public function deleteNotification()
    {
        $data = [
            'code'     => 'empty_code',
            'message'  => 'Empty Message'
        ];

        $data_json = $this->request->getJSON();
        
        // Token kontrolü
        if (empty($data_json->token)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Token eksik'
            ]);
        }

        $token = $data_json->token;
        $tokenModel = model('App\Models\TokenModel');
        
        $user_token = $tokenModel->where('token', $token)->findAll(1);

        if (count($user_token) == 0) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Geçerli token eksik'
            ]);
        }
        
        $userId = $user_token[0]['user_id'];
        
        // Bildirim ID kontrolü
        if (empty($data_json->notification_id)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Bildirim ID eksik'
            ]);
        }
        
        $notificationId = (int)$data_json->notification_id;
        
        // NotificationModel yükle
        $notificationModel = model('App\Models\NotificationModel');
        
        // Bildirimi sil
        $success = $notificationModel->deleteNotification($notificationId, $userId);
        
        if ($success) {
            return $this->respond([
                'code'     => 'success',
                'message'  => 'Bildirim başarıyla silindi',
                'notification_id' => $notificationId
            ]);
        } else {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Bildirim silinirken hata oluştu'
            ]);
        }
    }
    
    /**
     * Okunmamış bildirim sayısını getir
     */
    public function getUnreadCount()
    {
        // Token kontrolü
        if (empty($_GET['token'])) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Token eksik'
            ]);
        }

        $token = $_GET['token'];
        $tokenModel = model('App\Models\TokenModel');
        
        $user_token = $tokenModel->where('token', $token)->findAll(1);

        if (count($user_token) == 0) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Geçerli token eksik'
            ]);
        }
        
        $userId = $user_token[0]['user_id'];
        
        // NotificationModel yükle
        $notificationModel = model('App\Models\NotificationModel');
        
        // Okunmamış bildirim sayısını al
        $unreadCount = $notificationModel->getUnreadCount($userId);
        
        return $this->respond([
            'code'     => 'success',
            'unread_count' => $unreadCount
        ]);
    }
    
    
    /**
     * Bildirim gönder - Yönetici ve yetkili kullanıcılar için
     */
    public function send()
    {
        $data = [
            'code'     => 'empty_code',
            'message'  => 'Empty Message'
        ];

        $data_json = $this->request->getJSON();
        
        // Token kontrolü
        if (empty($data_json->token)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Token eksik'
            ]);
        }

        $token = $data_json->token;
        $tokenModel = model('App\Models\TokenModel');
        $userModel = model('App\Models\UserModel');
        
        $user_token = $tokenModel->where('token', $token)->findAll(1);

        if (count($user_token) == 0) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Geçerli token eksik'
            ]);
        }
        
        $userId = $user_token[0]['user_id'];
        
        // Yönetici yetkisi kontrolü
        // Sadece YÖNETİCİ rolüne sahip kullanıcılar bildirim gönderebilir (kullanıcı tipine göre kontrol)
        $user = $userModel->find($userId);
        
        if (!$user || $user['type'] !== 'ADMIN') {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Yetkisiz - bildirim göndermek için yönetici hakları gerekli'
            ]);
        }
        
        // Zorunlu alanları kontrol et
        if (empty($data_json->title) || empty($data_json->body)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Bildirim başlığı ve içeriği gerekli'
            ]);
        }
        
        // Hedef türünü kontrol et
        if (empty($data_json->target_type)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Hedef türü gerekli'
            ]);
        }
        
        // Bildirim servisini yükle
        $notificationService = service('NotificationService');
        $success = false;
        $response = [];
        
        // Ek verileri hazırla
        $data = !empty($data_json->data) && is_object($data_json->data) ? (array)$data_json->data : [];
        
        // Bildirim kaydı için veritabanı
        $notificationModel = model('App\Models\NotificationModel');
        
        // Seçenekleri ayarla
        $options = [];
        
        if (!empty($data_json->sound)) {
            $options['sound'] = $data_json->sound;
        }
        
        if (!empty($data_json->icon)) {
            $options['icon'] = $data_json->icon;
        }
        
        if (!empty($data_json->color)) {
            $options['color'] = $data_json->color;
        }
        
        if (!empty($data_json->click_action)) {
            $options['click_action'] = $data_json->click_action;
            $data['click_action'] = $data_json->click_action;
        }
        
        if (!empty($data_json->channel_id)) {
            $options['channel_id'] = $data_json->channel_id;
        }
        
        // Hedef tipine göre bildirim gönderimi
        switch ($data_json->target_type) {
            // Belirli bir kullanıcıya gönderim
            case 'user':
                if (empty($data_json->target_id)) {
                    return $this->respond([
                        'code'     => 'failed',
                        'message'  => 'Hedef kullanıcı ID gerekli'
                    ]);
                }
                
                $targetUserId = (int)$data_json->target_id;
                $success = $notificationService->sendToUser($targetUserId, $data_json->title, $data_json->body, $data, $options);
                
                // Bildirim kaydını veritabanına ekle
                $notificationData = [
                    'user_id' => $targetUserId,
                    'title' => $data_json->title,
                    'body' => $data_json->body,
                    'data' => json_encode($data),
                    'target' => 'user',
                    'status' => $success ? 'sent' : 'failed',
                    'created_by' => $userId
                ];
                
                $notificationModel->addNotification($notificationData);
                
                $response = [
                    'user_id' => $targetUserId,
                    'result' => $success
                ];
                break;
                
            // Birden fazla kullanıcıya gönderim
            case 'users':
                if (empty($data_json->target_ids) || !is_array($data_json->target_ids)) {
                    return $this->respond([
                        'code'     => 'failed',
                        'message'  => 'Hedef kullanıcı IDleri dizisi gerekli'
                    ]);
                }
                
                $targetUserIds = $data_json->target_ids;
                $results = $notificationService->sendToMultipleUsers($targetUserIds, $data_json->title, $data_json->body, $data, $options);
                
                $allSuccess = true;
                foreach ($results as $userId => $result) {
                    // Bildirim kaydını veritabanına ekle
                    $notificationData = [
                        'user_id' => $userId,
                        'title' => $data_json->title,
                        'body' => $data_json->body,
                        'data' => json_encode($data),
                        'target' => 'user',
                        'status' => $result ? 'sent' : 'failed',
                        'created_by' => $userId
                    ];
                    
                    $notificationModel->addNotification($notificationData);
                    
                    if (!$result) {
                        $allSuccess = false;
                    }
                }
                
                $success = $allSuccess;
                $response = [
                    'users' => $results
                ];
                break;
                
            // Konuya gönderim
            case 'topic':
                if (empty($data_json->topic)) {
                    return $this->respond([
                        'code'     => 'failed',
                        'message'  => 'Konu adı gerekli'
                    ]);
                }
                
                $topic = $data_json->topic;
                $success = $notificationService->sendToTopic($topic, $data_json->title, $data_json->body, $data, $options);
                
                // Bildirim kaydını veritabanına ekle
                $notificationData = [
                    'user_id' => null,
                    'title' => $data_json->title,
                    'body' => $data_json->body,
                    'data' => json_encode($data),
                    'target' => 'topic:' . $topic,
                    'status' => $success ? 'sent' : 'failed',
                    'created_by' => $userId
                ];
                
                $notificationModel->addNotification($notificationData);
                
                $response = [
                    'topic' => $topic,
                    'result' => $success
                ];
                break;
                
            // Tüm kullanıcılara gönderim (anonymous topic)
            case 'all_anonymous':
                $success = $notificationService->sendToTopic('anonymous', $data_json->title, $data_json->body, $data, $options);
                
                // Bildirim kaydını veritabanına ekle
                $notificationData = [
                    'user_id' => null,
                    'title' => $data_json->title,
                    'body' => $data_json->body,
                    'data' => json_encode($data),
                    'target' => 'topic:anonymous',
                    'status' => $success ? 'sent' : 'failed',
                    'created_by' => $userId
                ];
                
                $notificationModel->addNotification($notificationData);
                
                $response = [
                    'topic' => 'anonymous',
                    'result' => $success
                ];
                break;
                
            // Pazarlama izni olan tüm kullanıcılara gönderim
            case 'all_marketing':
                $success = $notificationService->sendToTopic('marketing', $data_json->title, $data_json->body, $data, $options);
                
                // Bildirim kaydını veritabanına ekle
                $notificationData = [
                    'user_id' => null,
                    'title' => $data_json->title,
                    'body' => $data_json->body,
                    'data' => json_encode($data),
                    'target' => 'topic:marketing',
                    'status' => $success ? 'sent' : 'failed',
                    'created_by' => $userId
                ];
                
                $notificationModel->addNotification($notificationData);
                
                $response = [
                    'topic' => 'marketing',
                    'result' => $success
                ];
                break;
                
            // Özel filtreli kullanıcılara gönderim (belirli kriterlere göre)
            case 'filtered':
                // Filtreleri hazırla
                $filters = [];
                
                if (!empty($data_json->filters) && is_object($data_json->filters)) {
                    if (isset($data_json->filters->platform)) {
                        $filters['platform'] = $data_json->filters->platform;
                    }
                    
                    if (isset($data_json->filters->marketing_consent)) {
                        $filters['marketing_consent'] = $data_json->filters->marketing_consent === true || 
                                                       $data_json->filters->marketing_consent === '1' || 
                                                       $data_json->filters->marketing_consent === 1;
                    }
                    
                    if (isset($data_json->filters->last_active_days)) {
                        $days = (int)$data_json->filters->last_active_days;
                        $filters['last_active_after'] = date('Y-m-d H:i:s', strtotime("-{$days} days"));
                    }
                }
                
                // Token limiti
                $limit = !empty($data_json->limit) ? (int)$data_json->limit : 1000;
                
                // Filtrelere göre token'ları al
                $tokens = $notificationService->getActiveTokens($filters, $limit);
                
                if (empty($tokens)) {
                    return $this->respond([
                        'code'     => 'failed',
                        'message'  => 'Belirtilen filtrelere uyan aktif cihaz bulunamadı'
                    ]);
                }
                
                // Bildirimleri gönder
                $success = $notificationService->sendToTokens($tokens, $data_json->title, $data_json->body, $data, $options);
                
                // Bildirim kaydını veritabanına ekle
                $notificationData = [
                    'user_id' => null,
                    'title' => $data_json->title,
                    'body' => $data_json->body,
                    'data' => json_encode($data),
                    'target' => 'filtered:' . json_encode($filters),
                    'status' => $success ? 'sent' : 'failed',
                    'created_by' => $userId
                ];
                
                $notificationModel->addNotification($notificationData);
                
                $response = [
                    'tokens_count' => count($tokens),
                    'result' => $success
                ];
                break;
                
            default:
                return $this->respond([
                    'code'     => 'failed',
                    'message'  => 'Geçersiz hedef türü'
                ]);
        }
        
        if ($success) {
            return $this->respond([
                'code'     => 'success',
                'message'  => 'Bildirim başarıyla gönderildi',
                'response' => $response
            ]);
        } else {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Bildirim gönderilirken hata oluştu',
                'response' => $response
            ]);
        }
    }
    
    /**
     * Bildirim geçmişini getir
     */
    public function history()
    {
        // Token kontrolü
        if (empty($_GET['token'])) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Token eksik'
            ]);
        }

        $token = $_GET['token'];
        $tokenModel = model('App\Models\TokenModel');
        
        $user_token = $tokenModel->where('token', $token)->findAll(1);

        if (count($user_token) == 0) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Geçerli token eksik'
            ]);
        }
        
        $userId = $user_token[0]['user_id'];
        
        // Sorgu parametreleri
        $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 50;
        $offset = isset($_GET['offset']) ? (int)$_GET['offset'] : 0;
        $targetType = isset($_GET['target_type']) ? $_GET['target_type'] : null;
        
        // NotificationModel yükle
        $notificationModel = model('App\Models\NotificationModel');
        
        // Bildirim geçmişini al
        $notifications = $notificationModel->getNotificationHistory($userId, $limit, $offset, $targetType);
        $totalCount = $notificationModel->getNotificationCount($userId, $targetType);
        
        return $this->respond([
            'code'     => 'success',
            'notifications' => $notifications,
            'total' => $totalCount,
            'limit' => $limit,
            'offset' => $offset,
        ]);
    }
    
  
    
    /**
     * Tüm bildirimleri okundu olarak işaretle
     */
    public function mark_all_read()
    {
        // Token kontrolü
        if (empty($_GET['token'])) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Token eksik'
            ]);
        }

        $token = $_GET['token'];
        $tokenModel = model('App\Models\TokenModel');
        
        $user_token = $tokenModel->where('token', $token)->findAll(1);

        if (count($user_token) == 0) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Geçerli token eksik'
            ]);
        }
        
        $userId = $user_token[0]['user_id'];
        
        // NotificationModel yükle
        $notificationModel = model('App\Models\NotificationModel');
        
        // Tüm bildirimleri okundu olarak işaretle
        $success = $notificationModel->markAllAsRead($userId);
        
        return $this->respond([
            'code'     => 'success',
            'message'  => 'Tüm bildirimler okundu olarak işaretlendi',
            'success'  => $success
        ]);
    }
    
    /**
     * Bildirimi sil
     */
    public function delete_notification()
    {
        $data = [
            'code'     => 'empty_code',
            'message'  => 'Empty Message'
        ];

        $data_json = $this->request->getJSON();
        
        // Token kontrolü
        if (empty($data_json->token)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Token eksik'
            ]);
        }

        $token = $data_json->token;
        $tokenModel = model('App\Models\TokenModel');
        
        $user_token = $tokenModel->where('token', $token)->findAll(1);

        if (count($user_token) == 0) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Geçerli token eksik'
            ]);
        }
        
        $userId = $user_token[0]['user_id'];
        
        // Bildirim ID kontrolü
        if (empty($data_json->notification_id)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Bildirim ID eksik'
            ]);
        }
        
        $notificationId = (int)$data_json->notification_id;
        
        // NotificationModel yükle
        $notificationModel = model('App\Models\NotificationModel');
        
        // Bildirimi sil
        $success = $notificationModel->deleteNotification($notificationId, $userId);
        
        if ($success) {
            return $this->respond([
                'code'     => 'success',
                'message'  => 'Bildirim başarıyla silindi',
                'notification_id' => $notificationId
            ]);
        } else {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Bildirim silinirken hata oluştu'
            ]);
        }
    }
    
    /**
     * Okunmamış bildirim sayısını getir
     */
    public function unread_count()
    {
        // Token kontrolü
        if (empty($_GET['token'])) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Token eksik'
            ]);
        }

        $token = $_GET['token'];
        $tokenModel = model('App\Models\TokenModel');
        
        $user_token = $tokenModel->where('token', $token)->findAll(1);

        if (count($user_token) == 0) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Geçerli token eksik'
            ]);
        }
        
        $userId = $user_token[0]['user_id'];
        
        // NotificationModel yükle
        $notificationModel = model('App\Models\NotificationModel');
        
        // Okunmamış bildirim sayısını al
        $unreadCount = $notificationModel->getUnreadCount($userId);
        
        return $this->respond([
            'code'     => 'success',
            'unread_count' => $unreadCount
        ]);
    }
    
    /**
     * FCM token güncelleme
     */
    public function update_token()
    {
        $data = [
            'code'     => 'empty_code',
            'message'  => 'Empty Message'
        ];

        $data_json = $this->request->getJSON();
        $deviceModel = model('App\Models\DeviceModel');

        // Zorunlu parametreleri kontrol et
        if (empty($data_json->device_uuid)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Device UUID eksik'
            ]);
        }

        if (empty($data_json->fcm_token)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'FCM token eksik'
            ]);
        }

        $fcmToken = $data_json->fcm_token;
        $deviceUuid = $data_json->device_uuid;

        // Ek verileri hazırla
        $additionalData = [
            'last_active' => date('Y-m-d H:i:s')
        ];

        // Kullanıcı kimlik bilgileri varsa doğrula
        if (!empty($data_json->user_id) && !empty($data_json->token)) {
            $tokenModel = model('App\Models\TokenModel');
            $token = $data_json->token;
            $user_token = $tokenModel->where('token', $token)->findAll(1);

            if (count($user_token) > 0 && $user_token[0]['user_id'] == $data_json->user_id) {
                $additionalData['user_id'] = $data_json->user_id;
            }
        }

        // Platform bilgisi varsa ekle
        if (!empty($data_json->platform)) {
            $additionalData['platform'] = $data_json->platform;
        }
        
        // Cihaz bilgileri varsa ekle
        if (!empty($data_json->device_name)) {
            $additionalData['device_name'] = $data_json->device_name;
        }
        
        if (!empty($data_json->device_model)) {
            $additionalData['device_model'] = $data_json->device_model;
        }
        
        if (!empty($data_json->os_version)) {
            $additionalData['os_version'] = $data_json->os_version;
        }

        // Push ve marketing izinleri varsa ekle
        if (isset($data_json->push_consent)) {
            $additionalData['push_consent'] = $data_json->push_consent === true || $data_json->push_consent === '1' ? 1 : 0;
        }
        
        if (isset($data_json->marketing_consent)) {
            $additionalData['marketing_consent'] = $data_json->marketing_consent === true || $data_json->marketing_consent === '1' ? 1 : 0;
        }

        try {
            // FCM Token'ı güncelle
            $success = $deviceModel->updateFcmToken($deviceUuid, $fcmToken, $additionalData);

            if ($success) {
                // Konu aboneliklerini güncelle (izinlere göre)
                // TODO: Topic aboneliklerini güncelleme işlemi
                
                return $this->respond([
                    'code'     => 'success',
                    'message'  => 'FCM token başarıyla güncellendi'
                ]);
            } else {
                return $this->respond([
                    'code'     => 'failed',
                    'message'  => 'FCM token güncellenirken hata oluştu'
                ]);
            }
        } catch (\Exception $e) {
            log_message('error', 'FCM token güncellenirken hata: ' . $e->getMessage());
            
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'FCM token güncellenirken hata: ' . $e->getMessage()
            ]);
        }
    }
    
    /**
     * Cihaz durumunu güncelle (aktif/pasif)
     */
    public function update_device_status()
    {
        $data = [
            'code'     => 'empty_code',
            'message'  => 'Empty Message'
        ];

        $data_json = $this->request->getJSON();
        $deviceModel = model('App\Models\DeviceModel');

        // Zorunlu parametreleri kontrol et
        if (empty($data_json->device_uuid)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Device UUID eksik'
            ]);
        }

        if (!isset($data_json->is_active)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Cihaz durumu (is_active) eksik'
            ]);
        }

        $deviceUuid = $data_json->device_uuid;
        $isActive = $data_json->is_active === true || $data_json->is_active === '1' || $data_json->is_active === 1;

        // Kullanıcı kimlik bilgileri varsa doğrula
        $userId = null;
        if (!empty($data_json->user_id) && !empty($data_json->token)) {
            $tokenModel = model('App\Models\TokenModel');
            $token = $data_json->token;
            $user_token = $tokenModel->where('token', $token)->findAll(1);

            if (count($user_token) > 0 && $user_token[0]['user_id'] == $data_json->user_id) {
                $userId = $data_json->user_id;
                
                // Güvenlik kontrolü - kullanıcı sadece kendi cihazlarını güncelleme hakkına sahip
                $device = $deviceModel->getDeviceByUuid($deviceUuid);
                if ($device && !empty($device['user_id']) && $device['user_id'] != $userId) {
                    return $this->respond([
                        'code'     => 'failed',
                        'message'  => 'Bu cihazı güncelleme yetkiniz yok'
                    ]);
                }
            }
        }

        try {
            // Cihaz durumunu güncelle
            $success = $deviceModel->setDeviceActive($deviceUuid, $isActive);

            if ($success) {
                return $this->respond([
                    'code'     => 'success',
                    'message'  => 'Cihaz durumu başarıyla güncellendi',
                    'device_uuid' => $deviceUuid,
                    'is_active' => $isActive
                ]);
            } else {
                return $this->respond([
                    'code'     => 'failed',
                    'message'  => 'Cihaz durumu güncellenirken hata oluştu'
                ]);
            }
        } catch (\Exception $e) {
            log_message('error', 'Cihaz durumu güncellenirken hata: ' . $e->getMessage());
            
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Cihaz durumu güncellenirken hata: ' . $e->getMessage()
            ]);
        }
    }
}