<?php

namespace App\Models;

use CodeIgniter\Model;

class DeviceModel extends Model
{
    protected $table = 'user_devices';
    protected $primaryKey = 'id';
    
    protected $useAutoIncrement = true;
    
    protected $returnType = 'array';
    protected $useSoftDeletes = false;
    
    protected $allowedFields = [
        'user_id',
        'email',
        'device_uuid',
        'fcmToken',
        'push_token',
        'platform',
        'device_name',
        'system_name',
        'system_version',
        'device_model',
        'brand',
        'device_brand',
        'manufacturer',
        'login_method',
        'android_version',
        'app_version',
        'app_build_number',
        'language',
        'country_code',
        'country',
         'lat',         
         'lon',      
        'region',
        'ip_address',
        'push_consent',
        'marketing_consent',
        'is_active',
        'last_active',
        'first_seen',
        'topic_subscriptions',
        'activation_count',
        'created_at',
        'updated_at',
    ];
    
    protected $useTimestamps = true;
    protected $createdField = 'created_at';
    protected $updatedField = 'updated_at';
    
    protected $validationRules = [
        'device_uuid' => 'permit_empty',
    ];
    
    /**
 * Kullanıcının gerçek IP adresini al - CloudFlare, proxy, load balancer desteği ile
 */
private function getRealUserIp() 
{
    $ip_keys = array(
        'HTTP_CF_CONNECTING_IP',      // CloudFlare
        'HTTP_X_REAL_IP',              // Nginx proxy
        'HTTP_CLIENT_IP',              // Shared Internet
        'HTTP_X_FORWARDED_FOR',        // Proxy
        'HTTP_X_FORWARDED',
        'HTTP_X_CLUSTER_CLIENT_IP',
        'HTTP_FORWARDED_FOR',
        'HTTP_FORWARDED',
        'REMOTE_ADDR'                  // Direct connection
    );
    
    foreach ($ip_keys as $key) {
        if (array_key_exists($key, $_SERVER) === true) {
            foreach (explode(',', $_SERVER[$key]) as $ip) {
                $ip = trim($ip);
                
                // Özel IP aralıklarını ve reserved IP'leri filtrele
                if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false) {
                    return $ip;
                }
            }
        }
    }
    
    // Fallback - en son çare olarak REMOTE_ADDR
    return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
}
    
    /**
     * Kurucu metod - tabloyu kontrol et
     */
    public function __construct()
    {
        parent::__construct();
        
        // Tablo kontrol et ve oluştur/güncelle
        $this->checkTableColumns();
    }
    
    /**
     * Tablo sütunlarını kontrol et ve eksikse oluştur
     */
    private function checkTableColumns()
    {
        $db = \Config\Database::connect();
        $forge = \Config\Database::forge();
        
        // Tablo var mı kontrol et
        $tables = $db->listTables();
        
        if (!in_array($this->table, $tables)) {
            // Tablo yoksa oluştur
            $forge->addField([
                'id' => [
                    'type' => 'INT',
                    'constraint' => 11,
                    'unsigned' => true,
                    'auto_increment' => true
                ],
                'user_id' => [
                    'type' => 'INT',
                    'constraint' => 11,
                    'null' => true
                ],
                'email' => [
                    'type' => 'VARCHAR',
                    'constraint' => 255,
                    'null' => true
                ],
                'device_uuid' => [
                    'type' => 'VARCHAR',
                    'constraint' => 255,
                    'null' => true
                ],
                'fcmToken' => [
                    'type' => 'VARCHAR',
                    'constraint' => 255,
                    'null' => true
                ],
                'push_token' => [
                    'type' => 'VARCHAR',
                    'constraint' => 255,
                    'null' => true
                ],
                'platform' => [
                    'type' => 'VARCHAR',
                    'constraint' => 20,
                    'null' => true
                ],
                'device_name' => [
                    'type' => 'VARCHAR',
                    'constraint' => 255,
                    'null' => true
                ],
                'system_name' => [
                    'type' => 'VARCHAR',
                    'constraint' => 50,
                    'null' => true
                ],
                'system_version' => [
                    'type' => 'VARCHAR',
                    'constraint' => 50,
                    'null' => true
                ],
                'device_model' => [
                    'type' => 'VARCHAR',
                    'constraint' => 255,
                    'null' => true
                ],
                'brand' => [
                    'type' => 'VARCHAR',
                    'constraint' => 100,
                    'null' => true
                ],
                'device_brand' => [
                    'type' => 'VARCHAR',
                    'constraint' => 100,
                    'null' => true
                ],
                'manufacturer' => [
                    'type' => 'VARCHAR',
                    'constraint' => 100,
                    'null' => true
                ],
                'android_version' => [
                    'type' => 'VARCHAR',
                    'constraint' => 50,
                    'null' => true
                ],
                'app_version' => [
                    'type' => 'VARCHAR',
                    'constraint' => 50,
                    'null' => true
                ],
                'app_build_number' => [
                    'type' => 'VARCHAR',
                    'constraint' => 50,
                    'null' => true
                ],
                'language' => [
                    'type' => 'VARCHAR',
                    'constraint' => 10,
                    'null' => true
                ],
                'country_code' => [
                    'type' => 'VARCHAR',
                    'constraint' => 10,
                    'null' => true
                ],
                'country' => [
                    'type' => 'VARCHAR',
                    'constraint' => 100,
                    'null' => true
                ],
                'region' => [
                    'type' => 'VARCHAR',
                    'constraint' => 100,
                    'null' => true
                ],
                'ip_address' => [
                    'type' => 'VARCHAR',
                    'constraint' => 45,
                    'null' => true
                ],
                'push_consent' => [
                    'type' => 'TINYINT',
                    'constraint' => 1,
                    'default' => 0
                ],
                'marketing_consent' => [
                    'type' => 'TINYINT',
                    'constraint' => 1,
                    'default' => 0
                ],
                'is_active' => [
                    'type' => 'TINYINT',
                    'constraint' => 1,
                    'default' => 1
                ],
                'last_active' => [
                    'type' => 'DATETIME',
                    'null' => true
                ],
                'first_seen' => [
                    'type' => 'DATETIME',
                    'null' => true
                ],
                'topic_subscriptions' => [
                    'type' => 'TEXT',
                    'null' => true
                ],
                'activation_count' => [
                    'type' => 'INT',
                    'constraint' => 11,
                    'default' => 1
                ],
                'created_at' => [
                    'type' => 'DATETIME',
                    'null' => true
                ],
                'updated_at' => [
                    'type' => 'DATETIME',
                    'null' => true
                ]
            ]);
            
            $forge->addKey('id', true);
            $forge->addKey('user_id');
            $forge->addKey('device_uuid');
            $forge->addKey('fcmToken');
            
            $forge->createTable($this->table);
            
            log_message('info', "Tablo '{$this->table}' oluşturuldu");
        } else {
            // Tablo varsa, eksik sütunları kontrol et
            $existingColumns = $db->getFieldNames($this->table);
            
            // Eksik sütunlar için SQL sorguları
            $columnChecks = [
                'device_uuid' => "ALTER TABLE {$this->table} ADD COLUMN device_uuid VARCHAR(255) NULL AFTER email",
                'fcmToken' => "ALTER TABLE {$this->table} ADD COLUMN fcmToken VARCHAR(255) NULL AFTER device_uuid",
                'push_token' => "ALTER TABLE {$this->table} ADD COLUMN push_token VARCHAR(255) NULL AFTER fcmToken",
                'platform' => "ALTER TABLE {$this->table} ADD COLUMN platform VARCHAR(20) NULL AFTER push_token",
                'login_method' => "ALTER TABLE {$this->table} ADD COLUMN login_method ENUM('email', 'google', 'apple', 'facebook', 'unknown') DEFAULT 'unknown' AFTER email", // ✅ YENİ
  
                'device_name' => "ALTER TABLE {$this->table} ADD COLUMN device_name VARCHAR(255) NULL AFTER platform",
                'system_name' => "ALTER TABLE {$this->table} ADD COLUMN system_name VARCHAR(50) NULL AFTER device_name",
                'system_version' => "ALTER TABLE {$this->table} ADD COLUMN system_version VARCHAR(50) NULL AFTER system_name",
                'device_model' => "ALTER TABLE {$this->table} ADD COLUMN device_model VARCHAR(255) NULL AFTER system_version",
                'brand' => "ALTER TABLE {$this->table} ADD COLUMN brand VARCHAR(100) NULL AFTER device_model",
                'device_brand' => "ALTER TABLE {$this->table} ADD COLUMN device_brand VARCHAR(100) NULL AFTER brand",
                'manufacturer' => "ALTER TABLE {$this->table} ADD COLUMN manufacturer VARCHAR(100) NULL AFTER device_brand",
                'android_version' => "ALTER TABLE {$this->table} ADD COLUMN android_version VARCHAR(50) NULL AFTER manufacturer",
                'app_version' => "ALTER TABLE {$this->table} ADD COLUMN app_version VARCHAR(50) NULL AFTER android_version",
                'app_build_number' => "ALTER TABLE {$this->table} ADD COLUMN app_build_number VARCHAR(50) NULL AFTER app_version",
                'language' => "ALTER TABLE {$this->table} ADD COLUMN language VARCHAR(10) NULL AFTER app_build_number",
                'country_code' => "ALTER TABLE {$this->table} ADD COLUMN country_code VARCHAR(10) NULL AFTER language",
                'country' => "ALTER TABLE {$this->table} ADD COLUMN country VARCHAR(100) NULL AFTER country_code",
                'region' => "ALTER TABLE {$this->table} ADD COLUMN region VARCHAR(100) NULL AFTER country",
                'ip_address' => "ALTER TABLE {$this->table} ADD COLUMN ip_address VARCHAR(45) NULL AFTER region",
                'push_consent' => "ALTER TABLE {$this->table} ADD COLUMN push_consent TINYINT(1) NOT NULL DEFAULT 0 AFTER ip_address",
                'marketing_consent' => "ALTER TABLE {$this->table} ADD COLUMN marketing_consent TINYINT(1) NOT NULL DEFAULT 0 AFTER push_consent",
                'is_active' => "ALTER TABLE {$this->table} ADD COLUMN is_active TINYINT(1) NOT NULL DEFAULT 1 AFTER marketing_consent",
                'last_active' => "ALTER TABLE {$this->table} ADD COLUMN last_active DATETIME NULL AFTER is_active",
                'first_seen' => "ALTER TABLE {$this->table} ADD COLUMN first_seen DATETIME NULL AFTER last_active",
                'topic_subscriptions' => "ALTER TABLE {$this->table} ADD COLUMN topic_subscriptions TEXT NULL AFTER first_seen",
                'activation_count' => "ALTER TABLE {$this->table} ADD COLUMN activation_count INT(11) NOT NULL DEFAULT 1 AFTER topic_subscriptions",
            ];
            
            // Eksik sütunları ekle
            foreach ($columnChecks as $column => $sql) {
                if (!in_array($column, $existingColumns)) {
                    try {
                        $db->query($sql);
                        log_message('info', "Sütun '{$column}' eklendi, tablo: {$this->table}");
                    } catch (\Exception $e) {
                        log_message('error', "Sütun '{$column}' eklenirken hata: " . $e->getMessage());
                    }
                }
            }
            
            // NULL değerleri güncelle
            try {
                $db->query("UPDATE {$this->table} SET is_active = 1 WHERE is_active IS NULL");
                $db->query("UPDATE {$this->table} SET push_consent = 0 WHERE push_consent IS NULL");
                $db->query("UPDATE {$this->table} SET marketing_consent = 0 WHERE marketing_consent IS NULL");
                $db->query("UPDATE {$this->table} SET topic_subscriptions = '[]' WHERE topic_subscriptions IS NULL");
                $db->query("UPDATE {$this->table} SET activation_count = 1 WHERE activation_count IS NULL");
                $db->query("UPDATE {$this->table} SET push_token = fcmToken WHERE fcmToken IS NOT NULL AND push_token IS NULL");
                $db->query("UPDATE {$this->table} SET fcmToken = push_token WHERE push_token IS NOT NULL AND fcmToken IS NULL");
            } catch (\Exception $e) {
                log_message('error', "NULL değerleri güncellenirken hata: " . $e->getMessage());
            }
            
            // Index kontrolü ve ekleme
            try {
                // Check if indexes already exist
                $indexExists = false;
                $indexQuery = $db->query("SHOW INDEX FROM {$this->table} WHERE Key_name = 'idx_device_uuid'");
                $indexResults = $indexQuery->getResultArray();
                if (empty($indexResults)) {
                    $db->query("ALTER TABLE {$this->table} ADD INDEX idx_device_uuid (device_uuid)");
                    log_message('info', "Index 'idx_device_uuid' created for table {$this->table}");
                }

                $indexQuery = $db->query("SHOW INDEX FROM {$this->table} WHERE Key_name = 'idx_fcm_token'");
                $indexResults = $indexQuery->getResultArray();
                if (empty($indexResults)) {
                    $db->query("ALTER TABLE {$this->table} ADD INDEX idx_fcm_token (fcmToken)");
                    log_message('info', "Index 'idx_fcm_token' created for table {$this->table}");
                }

                $indexQuery = $db->query("SHOW INDEX FROM {$this->table} WHERE Key_name = 'idx_user_id'");
                $indexResults = $indexQuery->getResultArray();
                if (empty($indexResults)) {
                    $db->query("ALTER TABLE {$this->table} ADD INDEX idx_user_id (user_id)");
                    log_message('info', "Index 'idx_user_id' created for table {$this->table}");
                }

                // Yeni: Kombinasyon index ekle
                $indexQuery = $db->query("SHOW INDEX FROM {$this->table} WHERE Key_name = 'idx_user_device'");
                $indexResults = $indexQuery->getResultArray();
                if (empty($indexResults)) {
                    $db->query("ALTER TABLE {$this->table} ADD INDEX idx_user_device (user_id, device_uuid)");
                    log_message('info', "Index 'idx_user_device' created for table {$this->table}");
                }
            } catch (\Exception $e) {
                // Log the error but continue execution
                log_message('error', "Error checking or creating indexes: " . $e->getMessage());
            }
        }
    }
    

    
    /**
     * Cihazı kaydet veya güncelle - DÜZELTME: Her kullanıcı girişinde yeni kayıt
     * 
     * @param array $deviceData Cihaz verileri
     * @return bool|int ID veya başarısızsa false
     */
    public function saveDevice($deviceData)
    {
        // Cihaz UUID kontrolü
        if (empty($deviceData['device_uuid'])) {
            log_message('warning', 'Cihaz UUID olmadan kayıt isteği yapıldı');
            return false;
        }
        
        // ✅ DÜZELTME: Her zaman gerçek IP'yi al
    $deviceData['ip_address'] = $this->getRealUserIp();
        
        // FCM Token'ı push_token olarak da kaydet (uyumluluk için)
        if (!empty($deviceData['fcmToken']) && empty($deviceData['push_token'])) {
            $deviceData['push_token'] = $deviceData['fcmToken'];
        } else if (empty($deviceData['fcmToken']) && !empty($deviceData['push_token'])) {
            $deviceData['fcmToken'] = $deviceData['push_token'];
        }
        
        // ÖNEMLİ: Kullanıcı ID varsa, aynı cihazda farklı kullanıcı girişi olup olmadığını kontrol et
        $existingDevice = null;
        
        if (!empty($deviceData['user_id'])) {
            // Aynı cihaz UUID ve kullanıcı ID kombinasyonunu ara
            $existingDevice = $this->where('device_uuid', $deviceData['device_uuid'])
                                   ->where('user_id', $deviceData['user_id'])
                                   ->orderBy('last_active', 'DESC')
                                   ->first();
            
            log_message('debug', 'Cihaz kontrolü - UUID: ' . $deviceData['device_uuid'] . 
                ', User ID: ' . $deviceData['user_id'] . 
                  ', IP: ' . $deviceData['ip_address'] . // ✅ IP'yi logla
                ', Mevcut kayıt: ' . ($existingDevice ? 'Evet' : 'Hayır'));
        } else {
            // Kullanıcı ID yoksa sadece device_uuid ile ara
            $existingDevice = $this->where('device_uuid', $deviceData['device_uuid'])
                                   ->where('user_id IS NULL', NULL, FALSE)
                                   ->orderBy('last_active', 'DESC')
                                   ->first();
        }
        
        if ($existingDevice) {
            // Aynı kullanıcı ve aynı cihaz - güncelle
            $deviceData['last_active'] = date('Y-m-d H:i:s');
            
            // Güncellenen alanları belirle
            $updateData = [];
            foreach ($deviceData as $key => $value) {
                if (in_array($key, $this->allowedFields)) {
                    $updateData[$key] = $value;
                }
            }
            
            // Aktivasyon sayısını artır
            if (!isset($updateData['activation_count'])) {
                $updateData['activation_count'] = ($existingDevice['activation_count'] ?? 0) + 1;
            }
            
            // Cihaz aktif olarak ayarla
            if (!isset($updateData['is_active'])) {
                $updateData['is_active'] = 1;
            }
            
            log_message('debug', 'Mevcut cihaz güncelleniyor - ID: ' . $existingDevice['id'] . 
                ', UUID: ' . $deviceData['device_uuid'] . 
                  ', IP: ' . $deviceData['ip_address'] . // ✅ IP'yi logla
                ', User ID: ' . ($deviceData['user_id'] ?? 'NULL'));
            
            return $this->update($existingDevice['id'], $updateData);
        } else {
            // Farklı kullanıcı girişi veya yeni cihaz - YENİ KAYIT OLUŞTUR
            $insertData = [];
            foreach ($deviceData as $key => $value) {
                if (in_array($key, $this->allowedFields)) {
                    $insertData[$key] = $value;
                }
            }
            
            // İlk görülme ve son aktivite tarihlerini ayarla
            if (!isset($insertData['first_seen'])) {
                $insertData['first_seen'] = date('Y-m-d H:i:s');
            }
            
            if (!isset($insertData['last_active'])) {
                $insertData['last_active'] = date('Y-m-d H:i:s');
            }
            
            // Aktivasyon sayısını ayarla
            if (!isset($insertData['activation_count'])) {
                $insertData['activation_count'] = 1;
            }
            
            // Aktif olarak işaretle
            if (!isset($insertData['is_active'])) {
                $insertData['is_active'] = 1;
            }
            
            // İzin durumlarını ayarla
            if (!isset($insertData['push_consent'])) {
                $insertData['push_consent'] = 0;
            }
            
            if (!isset($insertData['marketing_consent'])) {
                $insertData['marketing_consent'] = 0;
            }
            
            // Konu abonelikleri
            if (!isset($insertData['topic_subscriptions'])) {
                $topics = [];
                if (!empty($insertData['user_id'])) {
                    $topics[] = 'user_' . $insertData['user_id'];
                } else {
                    $topics[] = 'anonymous';
                }
                
                if (isset($insertData['marketing_consent']) && $insertData['marketing_consent'] == 1) {
                    $topics[] = 'marketing';
                }
                
                $insertData['topic_subscriptions'] = json_encode($topics);
            }
            
            log_message('info', 'YENİ cihaz kaydı oluşturuluyor - UUID: ' . $deviceData['device_uuid'] . 
                ', User ID: ' . ($insertData['user_id'] ?? 'NULL') . 
                  ', IP: ' . $insertData['ip_address'] . // ✅ IP'yi logla
                ', FCM Token: ' . ($insertData['fcmToken'] ?? 'NULL'));
            
            return $this->insert($insertData);
        }
    }
    
    /**
     * FCM token'ı güncelle - DÜZELTME: Her kullanıcı için ayrı kayıt
     * 
     * @param string $deviceUuid Cihaz UUID'si
     * @param string $token FCM token
     * @param array $additionalData Ek veriler
     * @return bool Başarı durumu
     */
    public function updateFcmToken($deviceUuid, $token, $additionalData = [])
    {
        // Boş token kontrolü
        if (empty($token)) {
            log_message('warning', 'Boş token değeri ile FCM token güncellemesi denendi');
            return false;
        }
        
        // Boş UUID kontrolü
        if (empty($deviceUuid)) {
            log_message('warning', 'Boş UUID ile FCM token güncellemesi denendi');
            return false;
        }
        
        log_message('info', "FCM token güncelleniyor - UUID: $deviceUuid, Token: $token");
        
        if (!empty($additionalData)) {
            log_message('debug', "Ek veriler: " . json_encode($additionalData, JSON_UNESCAPED_UNICODE));
        }
        
       // ✅ DÜZELTME: Her zaman gerçek IP'yi al
    $additionalData['ip_address'] = $this->getRealUserIp();
        
        // Kullanıcı ID varsa, kullanıcı-cihaz kombinasyonunu kontrol et
        $device = null;
        
         if (!empty($additionalData)) {
        log_message('debug', "Ek veriler: " . json_encode($additionalData, JSON_UNESCAPED_UNICODE));
        log_message('debug', "IP Adresi: " . $additionalData['ip_address']); // ✅ IP'yi özellikle logla
    }
        
        if (!empty($additionalData['user_id'])) {
            // Aynı kullanıcı ve aynı cihaz kombinasyonunu ara
            $device = $this->where('device_uuid', $deviceUuid)
                          ->where('user_id', $additionalData['user_id'])
                          ->orderBy('last_active', 'DESC')
                          ->first();
            
            log_message('debug', "Kullanıcı-cihaz kombinasyonu aranıyor - UUID: $deviceUuid, User ID: " . 
                $additionalData['user_id'] . ", Bulundu: " . ($device ? 'Evet' : 'Hayır'));
        } else {
            // Kullanıcı ID yoksa, sadece device UUID ile ara
            $device = $this->where('device_uuid', $deviceUuid)
                          ->where('user_id IS NULL', NULL, FALSE)
                          ->orderBy('last_active', 'DESC')
                          ->first();
        }
        
        if ($device) {
            // Mevcut kaydı güncelle
            $data = [
                'fcmToken' => $token,
                'push_token' => $token,
                'last_active' => date('Y-m-d H:i:s')
            ];
            
            // Ek verileri ekle
            if (!empty($additionalData)) {
                foreach ($additionalData as $key => $value) {
                    if (in_array($key, $this->allowedFields)) {
                        $data[$key] = $value;
                    }
                }
            }
            
            // Aktivasyon sayısını artır
            if (!isset($data['activation_count'])) {
                $data['activation_count'] = ($device['activation_count'] ?? 0) + 1;
            }
            
            // Cihaz durumunu aktif olarak ayarla
            if (!isset($data['is_active'])) {
                $data['is_active'] = 1;
            }
            
            log_message('debug', "Mevcut kayıt güncelleniyor - ID: {$device['id']}, UUID: $deviceUuid");
            
            return $this->update($device['id'], $data);
        } else {
            // Yeni kayıt oluştur
            $data = [
                'device_uuid' => $deviceUuid,
                'fcmToken' => $token,
                'push_token' => $token,
                'first_seen' => date('Y-m-d H:i:s'),
                'last_active' => date('Y-m-d H:i:s'),
                'activation_count' => 1,
                'is_active' => 1,
                'platform' => $additionalData['platform'] ?? 'unknown',
            ];
            
            // Ek verileri ekle
            if (!empty($additionalData)) {
                foreach ($additionalData as $key => $value) {
                    if (in_array($key, $this->allowedFields) && !isset($data[$key])) {
                        $data[$key] = $value;
                    }
                }
            }
            
            // İzin durumlarını varsayılan olarak ayarla
            if (!isset($data['push_consent'])) {
                $data['push_consent'] = isset($additionalData['push_consent']) ? 
                                      (int)$additionalData['push_consent'] : 0;
            }
            
            if (!isset($data['marketing_consent'])) {
                $data['marketing_consent'] = isset($additionalData['marketing_consent']) ? 
                                          (int)$additionalData['marketing_consent'] : 0;
            }
            
            // Konu abonelikleri
            $topics = [];
            if (!empty($data['user_id'])) {
                $topics[] = 'user_' . $data['user_id'];
            } else {
                $topics[] = 'anonymous';
            }
            
            if (isset($data['marketing_consent']) && $data['marketing_consent'] == 1) {
                $topics[] = 'marketing';
            }
            
            $data['topic_subscriptions'] = json_encode($topics);
            
            log_message('info', "YENİ FCM token kaydı oluşturuluyor - UUID: $deviceUuid, User ID: " . 
                ($data['user_id'] ?? 'NULL'));
            
            $insertId = $this->insert($data);
            return $insertId ? true : false;
        }
    }
    
    /**
     * Cihazı UUID'ye göre getir
     * 
     * @param string $deviceUuid Cihaz UUID'si
     * @return array|null Cihaz bilgileri veya bulunamazsa null
     */
    public function getDeviceByUuid($deviceUuid)
    {
        return $this->where('device_uuid', $deviceUuid)->first();
    }
    
    /**
     * Cihaz durumunu (aktif/pasif) ayarla
     * 
     * @param string $deviceUuid Cihaz UUID'si
     * @param bool $isActive Aktif mi
     * @return bool Başarı durumu
     */
    public function setDeviceActive($deviceUuid, $isActive = true)
    {
        $device = $this->where('device_uuid', $deviceUuid)->first();
        
        if ($device) {
            $data = [
                'is_active' => $isActive ? 1 : 0,
                'last_active' => date('Y-m-d H:i:s')
            ];
            
            if ($isActive) {
                $data['activation_count'] = ($device['activation_count'] ?? 0) + 1;
            }
            
            return $this->update($device['id'], $data);
        }
        
        return false;
    }
    
    /**
     * Konu aboneliğini güncelle
     * 
     * @param string $deviceUuid Cihaz UUID'si
     * @param string $topic Konu adı
     * @param bool $subscribe Abone olmak için true, abonelikten çıkmak için false
     * @return bool Başarı durumu
     */
    public function updateTopicSubscription($deviceUuid, $topic, $subscribe = true)
    {
        $device = $this->where('device_uuid', $deviceUuid)->first();
        
        if (!$device) {
            log_message('warning', "Konu aboneliği güncellenemiyor - Cihaz bulunamadı: $deviceUuid");
            return false;
        }
        
        // Mevcut abonelikleri al
        $subscriptions = [];
        if (!empty($device['topic_subscriptions'])) {
            try {
                $subscriptions = json_decode($device['topic_subscriptions'], true);
                
                // JSON diziye çevrilemezse boş dizi olarak kabul et
                if (!is_array($subscriptions)) {
                    $subscriptions = [];
                }
            } catch (\Exception $e) {
                log_message('error', 'Topic subscription JSON decode hatası: ' . $e->getMessage());
                $subscriptions = [];
            }
        }
        
        // Abonelik değişikliği
        $modified = false;
        
        // Konuyu ekle/çıkar
        if ($subscribe) {
            // Konu henüz eklenmemişse ekle
            if (!in_array($topic, $subscriptions)) {
                $subscriptions[] = $topic;
                $modified = true;
                log_message('debug', "Konu aboneliği eklendi - UUID: $deviceUuid, Konu: $topic");
            }
        } else {
            // Konu varsa çıkar
            $count = count($subscriptions);
            $subscriptions = array_values(array_filter($subscriptions, function($t) use ($topic) {
                return $t !== $topic;
            }));
            
            if (count($subscriptions) !== $count) {
                $modified = true;
                log_message('debug', "Konu aboneliği kaldırıldı - UUID: $deviceUuid, Konu: $topic");
            }
        }
        
        // Değişiklik yoksa güncelleme yapma
        if (!$modified) {
            return true;
        }
        
        // Güncelle
        return $this->update($device['id'], [
            'topic_subscriptions' => json_encode($subscriptions),
            'last_active' => date('Y-m-d H:i:s')
        ]);
    }
    
    /**
     * Kullanıcının cihazlarını getir
     * 
     * @param int $userId Kullanıcı ID'si
     * @param bool $onlyActive Sadece aktif cihazlar
     * @return array Kullanıcının cihazları
     */
    public function getUserDevices($userId, $onlyActive = true)
    {
        $query = $this->where('user_id', $userId);
        
        if ($onlyActive) {
            $query->where('is_active', 1);
        }
        
        return $query->orderBy('last_active', 'DESC')->findAll();
    }
    
    /**
     * Kullanıcının bildirim token'larını getir
     * 
     * @param int $userId Kullanıcı ID'si
     * @param bool $onlyActive Sadece aktif cihazlar
     * @return array Kullanıcının token'ları
     */
    public function getUserTokens($userId, $onlyActive = true)
    {
        $query = $this->where('user_id', $userId)
                      ->groupStart()
                        ->where('fcmToken !=', '')
                        ->where('fcmToken IS NOT NULL', NULL, FALSE)
                      ->groupEnd();
        
        if ($onlyActive) {
            $query->where('is_active', 1)
                  ->where('push_consent', 1);
        }
        
        return $query->findAll();
    }
    
    /**
     * Kullanıcıya ait tüm cihazların konu aboneliklerini güncelle
     * 
     * @param int $userId Kullanıcı ID'si
     * @param array $topicsToAdd Eklenecek konular
     * @param array $topicsToRemove Çıkarılacak konular
     * @return bool Başarı durumu
     */
    public function updateUserSubscriptions($userId, $topicsToAdd = [], $topicsToRemove = [])
    {
        $devices = $this->where('user_id', $userId)
                       ->where('is_active', 1)
                       ->findAll();
        
        if (empty($devices)) {
            log_message('info', "Kullanıcının aktif cihazı bulunamadı - UserId: $userId");
            return false;
        }
        
        $success = true;
        
        foreach ($devices as $device) {
            // Konuları ekle
            foreach ($topicsToAdd as $topic) {
                $result = $this->updateTopicSubscription($device['device_uuid'], $topic, true);
                if (!$result) {
                    $success = false;
                    log_message('error', "Konu aboneliği eklenemedi - UUID: {$device['device_uuid']}, Konu: $topic");
                }
            }
            
            // Konulardan çıkar
            foreach ($topicsToRemove as $topic) {
                $result = $this->updateTopicSubscription($device['device_uuid'], $topic, false);
                if (!$result) {
                    $success = false;
                    log_message('error', "Konu aboneliği kaldırılamadı - UUID: {$device['device_uuid']}, Konu: $topic");
                }
            }
        }
        
        return $success;
    }
    
    /**
     * Cihaz konu aboneliklerini getir
     * 
     * @param string $deviceUuid Cihaz UUID'si
     * @return array Konu abonelikleri
     */
    public function getDeviceTopics($deviceUuid)
    {
        $device = $this->where('device_uuid', $deviceUuid)->first();
        
        if (!$device || empty($device['topic_subscriptions'])) {
            return [];
        }
        
        try {
            $subscriptions = json_decode($device['topic_subscriptions'], true);
            
            // JSON diziye çevrilemezse boş dizi olarak kabul et
            if (!is_array($subscriptions)) {
                return [];
            }
            
            return $subscriptions;
        } catch (\Exception $e) {
            log_message('error', 'Topic subscription JSON decode hatası: ' . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Filtrelere göre aktif token'ları getir
     * 
     * @param array $filters Filtreler
     * @param int $limit Maksimum token sayısı
     * @return array Token'lar
     */
    public function getActiveTokensByFilters($filters = [], $limit = 1000)
    {
        $query = $this->select('fcmToken')
                      ->where('is_active', 1)
                      ->where('push_consent', 1)
                      ->where('fcmToken !=', '')
                      ->where('fcmToken IS NOT NULL', NULL, FALSE);
        
        // Platform filtresi
        if (isset($filters['platform']) && !empty($filters['platform'])) {
            if (is_array($filters['platform'])) {
                $query->whereIn('platform', $filters['platform']);
            } else {
                $query->where('platform', $filters['platform']);
            }
        }
        
        // Pazarlama izni filtresi
        if (isset($filters['marketing_consent'])) {
            $marketingConsent = $filters['marketing_consent'] ? 1 : 0;
            $query->where('marketing_consent', $marketingConsent);
        }
        
        // Ülke filtresi
        if (isset($filters['country_code']) && !empty($filters['country_code'])) {
            if (is_array($filters['country_code'])) {
                $query->whereIn('country_code', $filters['country_code']);
            } else {
                $query->where('country_code', $filters['country_code']);
            }
        }
        
        // Son aktif tarih filtresi
        if (isset($filters['last_active_after']) && !empty($filters['last_active_after'])) {
            $query->where('last_active >=', $filters['last_active_after']);
        }
        
        if (isset($filters['last_active_before']) && !empty($filters['last_active_before'])) {
            $query->where('last_active <=', $filters['last_active_before']);
        }
        
        // Uygulama sürümü filtresi
        if (isset($filters['app_version']) && !empty($filters['app_version'])) {
            if (is_array($filters['app_version'])) {
                $query->whereIn('app_version', $filters['app_version']);
            } else {
                $query->where('app_version', $filters['app_version']);
            }
        }
        
        // Kullanıcı filtresi
        if (isset($filters['user_id']) && !empty($filters['user_id'])) {
            if (is_array($filters['user_id'])) {
                $query->whereIn('user_id', $filters['user_id']);
            } else {
                $query->where('user_id', $filters['user_id']);
            }
        }
        
        // Cihaz modeli filtresi
        if (isset($filters['device_model']) && !empty($filters['device_model'])) {
            if (is_array($filters['device_model'])) {
                $query->whereIn('device_model', $filters['device_model']);
            } else {
                $query->where('device_model', $filters['device_model']);
            }
        }
        
        // Limit uygula ve sonuçları al
        $results = $query->limit($limit)->findAll();
        
        // Token'ları çıkar
        $tokens = [];
        foreach ($results as $result) {
            if (!empty($result['fcmToken'])) {
                $tokens[] = $result['fcmToken'];
            }
        }
        
        // Benzersiz token'ları döndür
        return array_values(array_unique($tokens));
    }
    
    /**
     * Email adresine göre cihaz bilgilerini getir
     * 
     * @param string $email Email adresi
     * @param bool $onlyActive Sadece aktif cihazlar
     * @return array Cihaz bilgileri
     */
    public function getDevicesByEmail($email, $onlyActive = true)
    {
        $query = $this->where('email', $email);
        
        if ($onlyActive) {
            $query->where('is_active', 1);
        }
        
        return $query->orderBy('last_active', 'DESC')->findAll();
    }
    
    /**
     * Cihaz sayısını getir
     * 
     * @param array $filters Filtreler
     * @return int Cihaz sayısı
     */
    public function getDeviceCount($filters = [])
    {
        $query = $this->builder();
        
        // Filtreler
        if (isset($filters['is_active'])) {
            $query->where('is_active', $filters['is_active'] ? 1 : 0);
        }
        
        if (isset($filters['user_id'])) {
            if (is_array($filters['user_id'])) {
                $query->whereIn('user_id', $filters['user_id']);
            } else {
                $query->where('user_id', $filters['user_id']);
            }
        }
        
        if (isset($filters['platform'])) {
            if (is_array($filters['platform'])) {
                $query->whereIn('platform', $filters['platform']);
            } else {
                $query->where('platform', $filters['platform']);
            }
        }
        
        if (isset($filters['country_code'])) {
            if (is_array($filters['country_code'])) {
                $query->whereIn('country_code', $filters['country_code']);
            } else {
                $query->where('country_code', $filters['country_code']);
            }
        }
        
        if (isset($filters['marketing_consent'])) {
            $query->where('marketing_consent', $filters['marketing_consent'] ? 1 : 0);
        }
        
        if (isset($filters['push_consent'])) {
            $query->where('push_consent', $filters['push_consent'] ? 1 : 0);
        }
        
        if (isset($filters['last_active_after'])) {
            $query->where('last_active >=', $filters['last_active_after']);
        }
        
        if (isset($filters['last_active_before'])) {
            $query->where('last_active <=', $filters['last_active_before']);
        }
        
        // NULL kontrolleri
        if (isset($filters['has_user_id'])) {
            if ($filters['has_user_id']) {
                $query->where('user_id IS NOT NULL', NULL, FALSE);
            } else {
                $query->where('user_id IS NULL', NULL, FALSE);
            }
        }
        
        if (isset($filters['has_email'])) {
            if ($filters['has_email']) {
                $query->where('email IS NOT NULL', NULL, FALSE)
                      ->where('email !=', '');
            } else {
                $query->groupStart()
                      ->where('email IS NULL', NULL, FALSE)
                      ->orWhere('email', '')
                      ->groupEnd();
            }
        }
        
        if (isset($filters['has_fcm_token'])) {
            if ($filters['has_fcm_token']) {
                $query->where('fcmToken IS NOT NULL', NULL, FALSE)
                      ->where('fcmToken !=', '');
            } else {
                $query->groupStart()
                      ->where('fcmToken IS NULL', NULL, FALSE)
                      ->orWhere('fcmToken', '')
                      ->groupEnd();
            }
        }
        
        // Cihaz sayısını döndür
        return $query->countAllResults();
    }
    
    /**
     * Cihazları filtrelerle getir
     * 
     * @param array $filters Filtreler
     * @param int $limit Limit
     * @param int $offset Offset
     * @param string $orderBy Sıralama alanı
     * @param string $orderDir Sıralama yönü
     * @return array Cihazlar
     */
    public function getDevicesWithFilters($filters = [], $limit = 50, $offset = 0, $orderBy = 'last_active', $orderDir = 'DESC')
    {
        $query = $this;
        
        // Filtreler
        if (isset($filters['is_active'])) {
            $query = $query->where('is_active', $filters['is_active'] ? 1 : 0);
        }
        
        if (isset($filters['user_id'])) {
            if (is_array($filters['user_id'])) {
                $query = $query->whereIn('user_id', $filters['user_id']);
            } else {
                $query = $query->where('user_id', $filters['user_id']);
            }
        }
        
        if (isset($filters['platform'])) {
            if (is_array($filters['platform'])) {
                $query = $query->whereIn('platform', $filters['platform']);
            } else {
                $query = $query->where('platform', $filters['platform']);
            }
        }
        
        if (isset($filters['country_code'])) {
            if (is_array($filters['country_code'])) {
                $query = $query->whereIn('country_code', $filters['country_code']);
            } else {
                $query = $query->where('country_code', $filters['country_code']);
            }
        }
        
        if (isset($filters['marketing_consent'])) {
            $query = $query->where('marketing_consent', $filters['marketing_consent'] ? 1 : 0);
        }
        
        if (isset($filters['push_consent'])) {
            $query = $query->where('push_consent', $filters['push_consent'] ? 1 : 0);
        }
        
        if (isset($filters['last_active_after'])) {
            $query = $query->where('last_active >=', $filters['last_active_after']);
        }
        
        if (isset($filters['last_active_before'])) {
            $query = $query->where('last_active <=', $filters['last_active_before']);
        }
        
        // NULL kontrolleri
        if (isset($filters['has_user_id'])) {
            if ($filters['has_user_id']) {
                $query = $query->where('user_id IS NOT NULL', NULL, FALSE);
            } else {
                $query = $query->where('user_id IS NULL', NULL, FALSE);
            }
        }
        
        if (isset($filters['has_email'])) {
            if ($filters['has_email']) {
                $query = $query->where('email IS NOT NULL', NULL, FALSE)
                              ->where('email !=', '');
            } else {
                $query = $query->groupStart()
                              ->where('email IS NULL', NULL, FALSE)
                              ->orWhere('email', '')
                              ->groupEnd();
            }
        }
        
        if (isset($filters['has_fcm_token'])) {
            if ($filters['has_fcm_token']) {
                $query = $query->where('fcmToken IS NOT NULL', NULL, FALSE)
                              ->where('fcmToken !=', '');
            } else {
                $query = $query->groupStart()
                              ->where('fcmToken IS NULL', NULL, FALSE)
                              ->orWhere('fcmToken', '')
                              ->groupEnd();
            }
        }
        
        // Metin arama
        if (isset($filters['search']) && !empty($filters['search'])) {
            $search = trim($filters['search']);
            $query = $query->groupStart()
                          ->like('device_uuid', $search)
                          ->orLike('email', $search)
                          ->orLike('device_name', $search)
                          ->orLike('device_model', $search)
                          ->orLike('brand', $search)
                          ->orLike('fcmToken', $search)
                          ->orLike('country', $search)
                          ->orLike('language', $search)
                          ->groupEnd();
        }
        
        // Sıralama
        if (!empty($orderBy) && in_array($orderBy, $this->allowedFields)) {
            $query = $query->orderBy($orderBy, $orderDir);
        } else {
            $query = $query->orderBy('last_active', 'DESC');
        }
        
        // Limit ve offset
        return $query->findAll($limit, $offset);
    }
    
    /**
     * Cihazı sil
     * 
     * @param string $deviceUuid Cihaz UUID'si
     * @return bool Başarı durumu
     */
    public function deleteDevice($deviceUuid)
    {
        $device = $this->where('device_uuid', $deviceUuid)->first();
        
        if (!$device) {
            log_message('warning', "Cihaz silinemedi - Bulunamadı: $deviceUuid");
            return false;
        }
        
        // Cihaz konularını temizle
        $db = \Config\Database::connect();
        try {
            $db->table('device_topic_subscriptions')
               ->where('device_uuid', $deviceUuid)
               ->delete();
        } catch (\Exception $e) {
            log_message('error', "Cihazın konu abonelikleri silinemedi: $deviceUuid, Hata: " . $e->getMessage());
        }
        
        return $this->delete($device['id']);
    }
    
    /**
     * Cihazı kullanıcıdan ayır
     * 
     * @param string $deviceUuid Cihaz UUID'si
     * @return bool Başarı durumu
     */
    public function detachDeviceFromUser($deviceUuid)
    {
        $device = $this->where('device_uuid', $deviceUuid)->first();
        
        if (!$device) {
            log_message('warning', "Cihaz kullanıcıdan ayrılamadı - Bulunamadı: $deviceUuid");
            return false;
        }
        
        // Kullanıcı bilgisi yoksa işlem yapma
        if (empty($device['user_id'])) {
            log_message('info', "Cihazın zaten kullanıcısı yok, işlem yapılmadı: $deviceUuid");
            return true;
        }
        
        // Kullanıcı ID'sini kaydet (log için)
        $oldUserId = $device['user_id'];
        
        // Cihazı kullanıcıdan ayır
        $data = [
            'user_id' => null,
            'push_consent' => 0,
            'marketing_consent' => 0,
            'last_active' => date('Y-m-d H:i:s')
        ];
        
        // Konu aboneliklerini güncelle
        $topics = ['anonymous']; // Anonim konuya abone ol
        $data['topic_subscriptions'] = json_encode($topics);
        
        // Kullanıcı konusundan ve pazarlama konusundan çık
        try {
            $this->updateTopicSubscription($deviceUuid, 'user_' . $oldUserId, false);
            $this->updateTopicSubscription($deviceUuid, 'marketing', false);
        } catch (\Exception $e) {
            log_message('error', "Konu abonelikleri güncellenemedi: $deviceUuid, Hata: " . $e->getMessage());
        }
        
        // Güncellemeyi logla
        log_message('info', "Cihaz kullanıcıdan ayrılıyor - UUID: $deviceUuid, Eski UserId: $oldUserId");
        
        return $this->update($device['id'], $data);
    }
    
    /**
     * İstatistik: Platformlara göre cihaz sayıları
     * 
     * @param bool $onlyActive Sadece aktif cihazlar
     * @return array Platform bazında cihaz sayıları
     */
    public function getDeviceCountByPlatform($onlyActive = true)
    {
        $query = $this->select('platform, COUNT(*) as count')
                      ->whereNotIn('platform', ['', 'unknown', null])
                      ->groupBy('platform');
        
        if ($onlyActive) {
            $query->where('is_active', 1);
        }
        
        $results = $query->findAll();
        
        // Sonuçları formatlı hale getir
        $stats = [];
        foreach ($results as $row) {
            $stats[$row['platform']] = (int)$row['count'];
        }
        
        return $stats;
    }
    
    /**
     * İstatistik: Ülkelere göre cihaz sayıları
     * 
     * @param bool $onlyActive Sadece aktif cihazlar
     * @return array Ülke bazında cihaz sayıları
     */
    public function getDeviceCountByCountry($onlyActive = true)
    {
        $query = $this->select('country_code, COUNT(*) as count')
                      ->whereNotIn('country_code', ['', null])
                      ->groupBy('country_code');
        
        if ($onlyActive) {
            $query->where('is_active', 1);
        }
        
        $results = $query->findAll();
        
        // Sonuçları formatlı hale getir
        $stats = [];
        foreach ($results as $row) {
            $stats[$row['country_code']] = (int)$row['count'];
        }
        
        return $stats;
    }
    
    /**
     * İstatistik: İzin durumlarına göre cihaz sayıları
     * 
     * @param bool $onlyActive Sadece aktif cihazlar
     * @return array İzin durumlarına göre cihaz sayıları
     */
    public function getDeviceCountByConsent($onlyActive = true)
    {
        $db = \Config\Database::connect();
        
        $whereClause = $onlyActive ? "WHERE is_active = 1" : "";
        
        $query = $db->query("
            SELECT
                COUNT(*) as total,
                SUM(CASE WHEN push_consent = 1 THEN 1 ELSE 0 END) as push_consent,
                SUM(CASE WHEN marketing_consent = 1 THEN 1 ELSE 0 END) as marketing_consent,
                SUM(CASE WHEN user_id IS NOT NULL THEN 1 ELSE 0 END) as logged_in,
                SUM(CASE WHEN fcmToken IS NOT NULL AND fcmToken != '' THEN 1 ELSE 0 END) as has_token
            FROM
                {$this->table}
            $whereClause
        ");
        
        $result = $query->getRowArray();
        
        // INT'e dönüştür
        foreach ($result as $key => $value) {
            $result[$key] = (int)$value;
        }
        
        return $result;
    }
    
    /**
     * İstatistik: Günlük aktivasyon sayıları
     * 
     * @param int $days Gün sayısı
     * @return array Günlük aktivasyon sayıları
     */
    public function getDailyActivations($days = 30)
    {
        $db = \Config\Database::connect();
        
        // Son n günlük aktivasyon sayıları
        $startDate = date('Y-m-d', strtotime("-$days days"));
        
        $query = $db->query("
            SELECT 
                DATE(created_at) as date,
                COUNT(*) as new_devices,
                SUM(CASE WHEN user_id IS NOT NULL THEN 1 ELSE 0 END) as logged_in_devices
            FROM 
                {$this->table}
            WHERE 
                created_at >= '$startDate'
            GROUP BY 
                DATE(created_at)
            ORDER BY 
                date ASC
        ");
        
        return $query->getResultArray();
    }
    
    /**
     * Konu aboneliklerine göre cihaz sayıları
     * 
     * @return array Konu aboneliklerine göre cihaz sayıları
     */
    public function getDeviceCountByTopic()
    {
        $db = \Config\Database::connect();
        
        $allDevices = $this->where('is_active', 1)->findAll();
        
        // Konulara göre sayıları hesapla
        $topicCounts = [
            'anonymous' => 0,
            'marketing' => 0,
        ];
        
        // Diğer konular için dinamik sayaçlar
        $userTopicCounts = [];
        $otherTopics = [];
        
        foreach ($allDevices as $device) {
            if (!empty($device['topic_subscriptions'])) {
                $topics = json_decode($device['topic_subscriptions'], true);
                
                if (is_array($topics)) {
                    foreach ($topics as $topic) {
                        if ($topic === 'anonymous') {
                            $topicCounts['anonymous']++;
                        } elseif ($topic === 'marketing') {
                            $topicCounts['marketing']++;
                        } elseif (strpos($topic, 'user_') === 0) {
                            // Kullanıcı konuları
                            if (!isset($userTopicCounts[$topic])) {
                                $userTopicCounts[$topic] = 0;
                            }
                            $userTopicCounts[$topic]++;
                        } else {
                            // Diğer konular
                            if (!isset($otherTopics[$topic])) {
                                $otherTopics[$topic] = 0;
                            }
                            $otherTopics[$topic]++;
                        }
                    }
                }
            }
        }
        
        // Kullanıcı konularını birleştir
        $topicCounts['user_topics_count'] = count($userTopicCounts);
        $topicCounts['user_devices_count'] = array_sum($userTopicCounts);
        
        // En popüler kullanıcı konuları
        arsort($userTopicCounts);
        $topicCounts['top_user_topics'] = array_slice($userTopicCounts, 0, 10, true);
        
        // Diğer konuları ekle
        if (!empty($otherTopics)) {
            $topicCounts['other_topics'] = $otherTopics;
        }
        
        return $topicCounts;
    }
    
  /**
     * Null değeri olan cihazları düzelt
     * 
     * @return bool Başarı durumu
     */
    public function fixNullValues()
    {
        $db = \Config\Database::connect();
        
        try {
            // Null veya boş değerleri düzelt
            $queries = [
                "UPDATE {$this->table} SET is_active = 1 WHERE is_active IS NULL",
                "UPDATE {$this->table} SET push_consent = 0 WHERE push_consent IS NULL",
                "UPDATE {$this->table} SET marketing_consent = 0 WHERE marketing_consent IS NULL",
                "UPDATE {$this->table} SET topic_subscriptions = '[]' WHERE topic_subscriptions IS NULL OR topic_subscriptions = ''",
                "UPDATE {$this->table} SET activation_count = 1 WHERE activation_count IS NULL OR activation_count = 0",
                "UPDATE {$this->table} SET push_token = fcmToken WHERE fcmToken IS NOT NULL AND push_token IS NULL",
                "UPDATE {$this->table} SET fcmToken = push_token WHERE push_token IS NOT NULL AND fcmToken IS NULL",
                "UPDATE {$this->table} SET last_active = created_at WHERE last_active IS NULL AND created_at IS NOT NULL",
                "UPDATE {$this->table} SET first_seen = created_at WHERE first_seen IS NULL AND created_at IS NOT NULL",
                "UPDATE {$this->table} SET created_at = NOW() WHERE created_at IS NULL",
                "UPDATE {$this->table} SET updated_at = NOW() WHERE updated_at IS NULL"
            ];
            
            foreach ($queries as $query) {
                $db->query($query);
            }
            
            // Topic subscriptions alanını güncelle
            $devices = $this->where("topic_subscriptions = '[]'", NULL, FALSE)
                          ->orWhere('topic_subscriptions IS NULL', NULL, FALSE)
                          ->findAll();
            
            foreach ($devices as $device) {
                $topics = [];
                
                // Kullanıcı ID varsa kullanıcı konusuna, yoksa anonim konuya abone ol
                if (!empty($device['user_id'])) {
                    $topics[] = 'user_' . $device['user_id'];
                } else {
                    $topics[] = 'anonymous';
                }
                
                // Pazarlama izni varsa, pazarlama konusuna abone ol
                if (!empty($device['marketing_consent']) && $device['marketing_consent'] == 1) {
                    $topics[] = 'marketing';
                }
                
                $this->update($device['id'], [
                    'topic_subscriptions' => json_encode($topics)
                ]);
            }
            
            return true;
        } catch (\Exception $e) {
            log_message('error', 'Null değerler düzeltilemedi: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * FCM token'ları var olan cihazları güncelle
     */
    public function updateMissingDeviceData()
    {
        try {
            // Boş UUID'li cihazları düzelt
            $devices = $this->where('device_uuid IS NULL', NULL, FALSE)
                          ->orWhere('device_uuid', '')
                          ->findAll();
            
            foreach ($devices as $device) {
                // UUID oluştur
                $uuid = \CodeIgniter\CodeIgniter::uuid();
                
                $this->update($device['id'], [
                    'device_uuid' => $uuid
                ]);
                
                log_message('info', "Cihaza UUID atandı - ID: {$device['id']}, UUID: $uuid");
            }
            
            // FCM token olan ancak push_token olmayan cihazları güncelle
            $this->where('fcmToken IS NOT NULL', NULL, FALSE)
                 ->where('fcmToken !=', '')
                 ->where('push_token IS NULL', NULL, FALSE)
                 ->orWhere('push_token', '')
                 ->set('push_token', 'fcmToken', FALSE)
                 ->update();
            
            // push_token olan ancak fcmToken olmayan cihazları güncelle
            $this->where('push_token IS NOT NULL', NULL, FALSE)
                 ->where('push_token !=', '')
                 ->where('fcmToken IS NULL', NULL, FALSE)
                 ->orWhere('fcmToken', '')
                 ->set('fcmToken', 'push_token', FALSE)
                 ->update();
                 
            return true;
        } catch (\Exception $e) {
            log_message('error', 'Cihaz verileri güncellenemedi: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Aynı kullanıcıya ait çoklu cihazları birleştir
     * 
     * @param int $userId Kullanıcı ID'si
     * @return bool Başarı durumu
     */
    public function mergeUserDevices($userId)
    {
        try {
            $devices = $this->where('user_id', $userId)->findAll();
            
            if (count($devices) <= 1) {
                return true; // Birleştirilecek cihaz yok
            }
            
            // En son aktif cihazı bul
            usort($devices, function($a, $b) {
                return strtotime($b['last_active'] ?? 0) - strtotime($a['last_active'] ?? 0);
            });
            
            $primaryDevice = $devices[0];
            $primaryTopics = json_decode($primaryDevice['topic_subscriptions'] ?? '[]', true);
            
            if (!is_array($primaryTopics)) {
                $primaryTopics = [];
            }
            
            // Diğer cihazları güncelle ve birleştir
            for ($i = 1; i < count($devices); $i++) {
                $device = $devices[$i];
                
                // Konu aboneliklerini birleştir
                $deviceTopics = json_decode($device['topic_subscriptions'] ?? '[]', true);
                
                if (is_array($deviceTopics)) {
                    foreach ($deviceTopics as $topic) {
                        if (!in_array($topic, $primaryTopics)) {
                            $primaryTopics[] = $topic;
                        }
                    }
                }
                
                // Birleştireceğimiz cihazı silmeyelim, sadece pasifleştirelim
                $this->update($device['id'], [
                    'is_active' => 0,
                    'push_consent' => 0,
                    'marketing_consent' => 0,
                    'updated_at' => date('Y-m-d H:i:s')
                ]);
            }
            
            // Ana cihazı güncelle
            $this->update($primaryDevice['id'], [
                'topic_subscriptions' => json_encode($primaryTopics),
                'updated_at' => date('Y-m-d H:i:s')
            ]);
            
            return true;
        } catch (\Exception $e) {
            log_message('error', "Kullanıcı cihazları birleştirilemedi - UserId: $userId, Hata: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Kullanıcı olmayan (user_id NULL) cihazları temizle
     * 
     * @param int $days Gün sayısı (bu günden daha eski olan cihazlar)
     * @return int Temizlenen cihaz sayısı
     */
    public function cleanupAnonymousDevices($days = 90)
    {
        try {
            $date = date('Y-m-d H:i:s', strtotime("-$days days"));
            
            // user_id NULL ve son aktivitesi belirli günden eski olan cihazları bul
            $devices = $this->where('user_id IS NULL', NULL, FALSE)
                          ->where('last_active <', $date)
                          ->findAll();
            
            $count = 0;
            foreach ($devices as $device) {
                // Cihazı sil
                $this->delete($device['id']);
                $count++;
                
                log_message('info', "Anonim cihaz temizlendi - ID: {$device['id']}, UUID: {$device['device_uuid']}");
            }
            
            return $count;
        } catch (\Exception $e) {
            log_message('error', 'Anonim cihazlar temizlenemedi: ' . $e->getMessage());
            return 0;
        }
    }
    
    /**
     * User_id NULL olan ama email'i aynı olan cihazları kullanıcıya bağla
     * 
     * @return int Güncellenen cihaz sayısı
     */
    public function linkOrphanDevices()
    {
        try {
            $db = \Config\Database::connect();
            
            // Email'i olan ama user_id'si olmayan cihazları bul
            $orphanDevices = $this->where('user_id IS NULL', NULL, FALSE)
                                ->where('email IS NOT NULL', NULL, FALSE)
                                ->where('email !=', '')
                                ->findAll();
            
            $count = 0;
            foreach ($orphanDevices as $device) {
                // Email'e göre kullanıcı ara
                $userModel = model('App\Models\UserModel');
                
                // Önce mail alanında ara
                $user = $userModel->where('mail', $device['email'])->first();
                
                // Mail alanında yoksa auth_gmail'de ara
                if (!$user) {
                    $user = $userModel->where('auth_gmail', $device['email'])->first();
                }
                
                // Kullanıcı bulunduysa cihazı güncelle
                if ($user) {
                    $this->update($device['id'], [
                        'user_id' => $user['id'],
                        'updated_at' => date('Y-m-d H:i:s')
                    ]);
                    
                    // Konu aboneliklerini güncelle
                    $this->updateTopicSubscription($device['device_uuid'], 'anonymous', false);
                    $this->updateTopicSubscription($device['device_uuid'], 'user_' . $user['id'], true);
                    
                    $count++;
                    
                    log_message('info', "Sahipsiz cihaz kullanıcıya bağlandı - DeviceID: {$device['id']}, Email: {$device['email']}, UserID: {$user['id']}");
                }
            }
            
            return $count;
        } catch (\Exception $e) {
            log_message('error', 'Sahipsiz cihazlar bağlanamadı: ' . $e->getMessage());
            return 0;
        }
    }
}