<?php

namespace App\Models;

use CodeIgniter\Model;

class ForgetMeModel extends Model
{
    protected $table = 'gdpr_forget_me_requests';
    protected $primaryKey = 'id';
    
    protected $useAutoIncrement = true;
    
    protected $returnType = 'array';
    protected $useSoftDeletes = false;
    
    protected $allowedFields = [
        'user_id',
        'reason',
        'device_uuid',
        'status',
        'reject_reason',
        'created_at',
        'processed_at',
        'processed_by',
        'ip_address',
        'user_agent'
    ];
    
    protected $useTimestamps = false;
    
    /**
     * Kurucu metod - tabloyu kontrol et
     */
    public function __construct()
    {
        parent::__construct();
        
        // Tablo kontrol et ve oluştur/güncelle
        $this->checkTableColumns();
    }
    
    /**
     * Tablo sütunlarını kontrol et ve eksikse oluştur
     */
    private function checkTableColumns()
    {
        $db = \Config\Database::connect();
        $forge = \Config\Database::forge();
        
        // Tablo var mı kontrol et
        $tables = $db->listTables();
        
        if (!in_array($this->table, $tables)) {
            // Tablo yoksa oluştur
            $forge->addField([
                'id' => [
                    'type' => 'INT',
                    'constraint' => 11,
                    'unsigned' => true,
                    'auto_increment' => true
                ],
                'user_id' => [
                    'type' => 'INT',
                    'constraint' => 11,
                    'null' => false
                ],
                'reason' => [
                    'type' => 'TEXT',
                    'null' => true
                ],
                'device_uuid' => [
                    'type' => 'VARCHAR',
                    'constraint' => 255,
                    'null' => true
                ],
                'status' => [
                    'type' => 'ENUM',
                    'constraint' => ['pending', 'approved', 'rejected'],
                    'default' => 'pending'
                ],
                'reject_reason' => [
                    'type' => 'TEXT',
                    'null' => true
                ],
                'created_at' => [
                    'type' => 'DATETIME',
                    'null' => true
                ],
                'processed_at' => [
                    'type' => 'DATETIME',
                    'null' => true
                ],
                'processed_by' => [
                    'type' => 'INT',
                    'constraint' => 11,
                    'null' => true
                ],
                'ip_address' => [
                    'type' => 'VARCHAR',
                    'constraint' => 45,
                    'null' => true
                ],
                'user_agent' => [
                    'type' => 'TEXT',
                    'null' => true
                ]
            ]);
            
            $forge->addKey('id', true);
            $forge->addKey('user_id');
            $forge->addKey('status');
            $forge->addKey('created_at');
            
            $forge->createTable($this->table);
            
            log_message('info', "Tablo '{$this->table}' oluşturuldu");
        }
    }
    
    /**
     * Kullanıcı bilgileri ile birlikte unutulma hakkı taleplerini getir
     * 
     * @param string|null $status Filtrelenecek durum (pending, approved, rejected)
     * @param int $limit Getirilebilecek maksimum talep sayısı
     * @param int $offset Sayfalama için başlangıç indeksi
     * @return array Talepler
     */
    public function getRequestsWithUserInfo($status = null, $limit = 100, $offset = 0)
    {
        $this->select('gdpr_forget_me_requests.*, users.name_surname, users.mail');
        $this->join('users', 'users.id = gdpr_forget_me_requests.user_id', 'left');
        
        if ($status !== null) {
            $this->where('gdpr_forget_me_requests.status', $status);
        }
        
        $this->orderBy('gdpr_forget_me_requests.created_at', 'DESC');
        
        return $this->findAll($limit, $offset);
    }
    
    /**
     * Kullanıcının daha önce unutulma hakkı talebi yapmış mı kontrol et
     * 
     * @param int $userId Kullanıcı ID
     * @return bool Talep var mı
     */
    public function hasExistingRequest($userId)
    {
        $existingRequests = $this->where('user_id', $userId)
                                 ->where('status', 'pending')
                                 ->countAllResults();
        
        return $existingRequests > 0;
    }
    
    /**
     * Belirli bir durumda olan talep sayısını getir
     * 
     * @param string $status Durum (pending, approved, rejected)
     * @return int Talep sayısı
     */
    public function getRequestCountByStatus($status)
    {
        return $this->where('status', $status)->countAllResults();
    }
    
    /**
     * İşlenmeyen (bekleyen) unutulma hakkı talepleri için yöneticilere hatırlatıcı e-posta gönder
     * 
     * @param int $daysThreshold Kaç gün önce oluşturulan talepler için hatırlatma gönderilecek
     * @return bool İşlem başarılı mı
     */
    public function sendReminderForPendingRequests($daysThreshold = 10)
    {
        $thresholdDate = date('Y-m-d H:i:s', strtotime("-$daysThreshold days"));
        
        $pendingRequests = $this->where('status', 'pending')
                                ->where('created_at <=', $thresholdDate)
                                ->findAll();
        
        if (empty($pendingRequests)) {
            return true; // Hatırlatılacak talep yok
        }
        
        $emailService = service('EmailService');
        $userModel = model('App\Models\UserModel');
        
        // Admin kullanıcıları bul
        $admins = $userModel->where('type', 'ADMIN')->findAll();
        
        if (empty($admins)) {
            return false; // Admin yok
        }
        
        $subject = "Hatırlatma: İşlenmemiş GDPR Unutulma Hakkı Talepleri";
        $message = "Merhaba,\n\n";
        $message .= "Aşağıdaki GDPR unutulma hakkı talepleri $daysThreshold günden uzun süredir işlenmemiştir:\n\n";
        
        foreach ($pendingRequests as $request) {
            $user = $userModel->find($request['user_id']);
            $userName = $user ? $user['name_surname'] : 'Bilinmeyen kullanıcı';
            $userEmail = $user ? $user['mail'] : 'Bilinmeyen email';
            
            $createdDate = date('d.m.Y H:i', strtotime($request['created_at']));
            
            $message .= "Talep ID: {$request['id']}\n";
            $message .= "Kullanıcı: $userName ($userEmail)\n";
            $message .= "Oluşturulma Tarihi: $createdDate\n";
            $message .= "Sebep: {$request['reason']}\n\n";
        }
        
        $message .= "Lütfen bu talepleri en kısa sürede işleme alınız. GDPR düzenlemeleri gereği, unutulma hakkı taleplerinin 30 gün içerisinde işleme alınması gerekmektedir.\n\n";
        $message .= "Saygılarımızla,\nSistem";
        
        $successCount = 0;
        
        foreach ($admins as $admin) {
            if (!empty($admin['mail'])) {
                $result = $emailService->sendEmail($admin['mail'], $subject, $message);
                if ($result) {
                    $successCount++;
                }
            }
        }
        
        return $successCount > 0;
    }
}